//////////////////////////////////////////////////////////////////////////
//                             countWords.c                             //
//                                                                      //
// Purpose                                                              //
// =======                                                              //
// The program searches through a text file searching for and counting  //
// the number of occurrences of a user-specified word. It reports the   //
// number of occurrences found, and the percentage that represents of   //
// the total number of words in the file.                               //
// A word is any sequence of letters and digits. Any uppercase letters  //
// are converted to lowercase.                                          //
// The program prints instructions for the user at its start.           //
//                                                                      //
// Author            Creation Date                                      //
// ======            =============                                      //
// P.A. Lee          12 December 1995                                   //
//                                                                      //
// Input/Output                                                         //
// ============                                                         //
// The user is requested to input the name of the file and the word(s)  //
// to be searched for.                                                  //
//                                                                      //
// Error messages are output if the specified file doesn't exist or     //
// for erroneous inputs. The number and percentage of occurrences are   //
// reported for each specified word.                                    //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <cctype>
#include <fstream>

using namespace std;

#include "error.h"
#include "Text.h"
#include "Word.h"

void testWord();

//////////////////////////////////////////////////////////////////////////
//                        Function Interfaces                           //
//////////////////////////////////////////////////////////////////////////

Text getFileName();
bool badFileName(Text name);
void showSomeLines(Text name);
void searchForWord(Text name, Word word, int& wordCount, int& totalCount);
void printProgressCount(int number);
bool getYesOrNo(Text message);

//////////////////////////////////////////////////////////////////////////
//                            Main program                              //
//////////////////////////////////////////////////////////////////////////

int main()
{
  Text fileName = "";
  
  cout << "Word Counting Program" << endl << "---------------------"     << endl
       << "You have to specify a file name for the file to be searched." << endl
       << "  A small text file named 'Test.txt' is provided."            << endl
       << "  This file provides some WWW addresses where electronic"     << endl
       << "  versions of copyright-free books are held, and you can get" << endl
       << "  hold of these books to conduct your own word searches"      << endl
       << endl;          
  
  while (fileName == "")
  {
    fileName = getFileName();
    if (badFileName(fileName))
    {
      cout << "** Sorry, but I can't read file '" << fileName << "'" << endl;
      fileName = "";
    }
  } // while (filename ...)
  
  cout << endl;
  if (getYesOrNo("Do you want to see some lines of the file"))
  {
    showSomeLines(fileName);
  }
  cout << endl << "Note that searches ignore the case of letters." << endl;
  while (getYesOrNo("Do you want to search for a word"))
  { 
    Text textLine;
    Word word             = Word("");  // Initialize to an empty word
    int  numberWords      = 0;
    int  totalNumberWords = 0;
    
    while (word == Word(""))           // Get a legal word from the user
    {
      cout << "Type the word that you wish to search for: ";
      getline(cin, textLine);         // Read whole line
      word = Word(textLine);           // Convert to a Word instance
      if (word == Word(""))
      {
        cout << "** That wasn't a legal word. Please try again." << endl;
      }
    }
    searchForWord(fileName, word, numberWords, totalNumberWords);
    cout << endl        << "The word '" << word << "' occurred "
         << numberWords << " times";
    if (numberWords != 0)
    {
      cout << " (" << double(numberWords) * 100.0/double(totalNumberWords)
           << "%)";
    }
    cout << " in a total of " << totalNumberWords << " words" << endl << endl;
   } // end while
    
  cout << "End of program" << endl;
  return 0;
} // end main()

//////////////////////////////////////////////////////////////////////////
//                        Function Definitions                          //
//////////////////////////////////////////////////////////////////////////

Text getFileName()
{
  Text name;
  
  cout << "Name of file to be searched: ";
  getline(cin, name);
  return name;
} // end getFileName()

bool badFileName(Text name)
{
  ifstream inFile;
  char ch;
  
  inFile.open(name);
  ch = inFile.peek();   // Try to look at the first character
  if (inFile.good())
  {
    return false;       // The file is OK
  }
  else
  {
    return true;        // There's a problem with the file
  }
} // end badFileName()

void showSomeLines(Text name)
{
  ifstream inFile;
  bool     keepGoing = true;
  
  inFile.open(name);
  while (keepGoing)
  {
    for (int countLines = 0; countLines < 10; countLines += 1)
    {
      Text line;
      
      getline(inFile, line);
      if (!inFile)
      {
        if (inFile.eof())
        {
          cout << "End of file reached" << endl;
          keepGoing = false;
          break;                  // Break out of for loop
        }
        Text message = "Error: reading file '" + name + "'";
        error(message);           // Terminates the program
      }
      cout << line << endl;
    }
    cout << endl;
    if (keepGoing && (getYesOrNo("Do you want to see 10 more lines") == false))
    {
      keepGoing = false;          // if user doesn't want more
    }
  } // end while (true)
  return;
} // end showSomeLines()

void searchForWord(Text name, Word word, int& wordCount, int& totalCount)
{
  const int PRINT_WHEN = 100;
  ifstream inFile;
  Word nextWord;
  
  inFile.open(name);
  inFile >> nextWord;
  wordCount  = 0;
  totalCount = 0;
  cout << "Searching ...          ";
  while (inFile.good())
  {
    if ((totalCount % PRINT_WHEN) == 0) 
    { 
      printProgressCount(totalCount);
    }  
    if (nextWord == word)
    {
      wordCount += 1;
    }
    totalCount += 1;    
    inFile >> nextWord;  
  } // end while (true)
  printProgressCount(totalCount);
  cout << endl;
  return; 
} // end searchForWord()

void printProgressCount(int number)
{                         
  const int MAX_SIZE   = 6;  // Maximum expected width of the count
  
  for (int i = 0; i < MAX_SIZE; i += 1)
  {
    cout << '\b';            // Backspace over previous output
  }
  cout.width(MAX_SIZE);
  cout << number;
  cout.flush();
  return;
} // end printProgressCount()


bool getYesOrNo(Text message)
{
  Text response;
  bool result;
  
  while (true)
  {
    cout << message << " (y/n): ";
    getline(cin, response);
    if ((response == "Y"  ) || (response == "y"  ) ||
        (response == "yes") || (response == "YES")   )
    {
      result = true;
      break;
    }
    else if ((response == "N" ) || (response == "n") ||
             (response == "no") || (response == "NO")  )
    {
      result = false;
      break;
    }
    cout << "You didn't type y or n. Please try again" << endl;
  } // end while (true)
  return result; 
} // end getYesOrNo()
