//////////////////////////////////////////////////////////////////////
//                               sieve.c                            //
//                                                                  //
// Purpose                                                          //
// =======                                                          //
// This program is a simple implementation of the sieve of          //
// Eratosthenes for finding prime numbers                           //
//                                                                  //
// Author            Creation Date                                  //
// ======            =============                                  //
// Eratosthenes      13 September 3BC                               //
//                                                                  //
// Input/Output                                                     //
// ============                                                     //
// Input:  The last natural number to be checked                    //
// Output: The prime numbers identified, and a graph of frequencies //
//            (if possible to draw)                                 //
//////////////////////////////////////////////////////////////////////

#include <iostream>
using namespace std;

#include "Array.h"
#include "Canvas.h"
#include "error.h"
#include "Text.h"

void fillPossibles(Array<bool>& possibles);
void cancelPossibles(Array<bool>& possibles);
void printPossibles(Array<bool> possibles);
void drawFrequencies(Array<bool> possibles);
Text intToText(int i);

int main()
{
  int         n;
  Array<bool> possibles;        // Create possibles

  cout << "Type the number to stop at: ";
  cin  >> n;
  if (n <= 1)
  {
    cout << "Error: number must be greater than 1" << endl;
    return -1;                  // Terminate the program
                                // with error flag
  }
  possibles.setSize(n);
  fillPossibles(possibles);     // Fill possibles
  cancelPossibles(possibles);   // Cancel non-primes in possibles
  printPossibles(possibles);    // Print out uncanceled elements
  drawFrequencies(possibles);   // Draw frequency graph

  return 0;                     // Terminate the program
} // end main()

void fillPossibles(Array<bool>& possibles)
{
  for (int initial = 0; initial < possibles.size(); initial += 1)
  {
    possibles(initial) = true;
  }
  return;
} // end fillPossibles()

void cancelPossibles(Array<bool>& possibles)
{                           // For each element in possibles
  for (int nextCheck = 2; nextCheck < possibles.size(); nextCheck += 1)
  {
    if (possibles(nextCheck))    // Has it been canceled?
    {                                // No: cancel in rest of possibles
      for (int cancel = nextCheck + nextCheck;  // Hop down array in
           cancel < possibles.size();           // nextCheck steps
           cancel += nextCheck)                 // canceling elements
      {
        possibles(cancel) = false;              // Set to CANCEL
      } // end for cancel
    } // end if possibles
  } // end for nextCheck
  return;
} // end cancelPossibles()

void printPossibles(Array<bool> possibles)
{
  int count = 0;
  cout << "The primes are:" << endl;
  for (int prime = 1; prime < possibles.size(); prime += 1)
  {
    if (possibles(prime))                // Has element been canceled?
    {
      cout << prime << ' ';                // No - print value
      count += 1;
    }
  }
  cout << endl << endl << "The total number was " 
       << count << endl;
  return;
} // end printPossibles()

void drawFrequencies(Array<bool> possibles)
{
  const int XMAX      = 200;               // Size of picture
  const int YMAX      = XMAX;
  const int MARGIN    = 1;
  const int BLOCKSIZE = 10;
  
  int roundUp = 0;
  int freq;
  Array<int> frequencies;

  if (possibles.size() % BLOCKSIZE != 0)    // Valid size?
  {
    roundUp = 1;   // To round up number of blocks to be drawn
  }
  
  frequencies.setSize((possibles.size() / BLOCKSIZE) + roundUp);

  const int XSCALE = XMAX / frequencies.size();  // Set scaling factors
  const int YSCALE = YMAX / BLOCKSIZE;
  
  // The columns must be at least one pixel wide for the
  //   frequency graph to be drawn. If XSCALE is zero they
  //   aren't, and either the window size or the BLOCKSIZE
  //   need to be increased for this range of numbers.
  
  if (XSCALE == 0) 
  { 
    cout << "Sorry: This range of numbers is too large to be plotted" << endl;
    return;
  }
  
  for (freq = 0; freq < frequencies.size(); freq += 1)
  {
    frequencies(freq) = 0;                // Initialize counts
  }
  
  for (int prime = 1; prime < possibles.size(); prime += 1)
  {
    if (possibles(prime))                // Is element prime?
    {                                       // Yes. Add 1 to
      frequencies(prime / BLOCKSIZE) += 1;  // block's count
    }
  }
   
  Canvas frequencyGraph;                 // Create Canvas
   
  // Set Canvas size and then draw the frequency graph
  frequencyGraph.setSize(XMAX + MARGIN, YMAX + MARGIN);  
  frequencyGraph.penDown();
  for (freq = 0; freq < frequencies.size(); freq += 1)
  {
    frequencyGraph.moveTo(freq * XSCALE, frequencies(freq) * YSCALE);
    frequencyGraph.moveTo((freq + 1) * XSCALE,
                          frequencies(freq) * YSCALE);
  }
  frequencyGraph.moveTo(frequencies.size() * XSCALE, 0);
  
  // Add a title
  Text graphTitle = "Plot of frequencies of primes";
  
  frequencyGraph.penUp();
  frequencyGraph.moveTo(20, YMAX - 20); 
  frequencyGraph.displayText(graphTitle);
  graphTitle = "in the range 1 to " + intToText(possibles.size());
  frequencyGraph.moveTo(20, YMAX - 40); 
  frequencyGraph.displayText(graphTitle);
  return;
} // end drawFrequencies()

            // Convert an integer to a string of characters
Text intToText(int i)
{
  Text message = "";
  
  // Generate digits, starting with the least significant
  while (i > 0)
  {
    char nextDigit = char((i % 10) + int('0'));
    
    message =  Text(nextDigit) + message;
    i /= 10;
  }
  return message;
} // end intToText()
