//////////////////////////////////////////////////////////////////////////
//                             Account.c                                //
//                                                                      //
// Purpose                                                              //
// =======                                                              //
// This file is the implementation for class Account.                   //
//                                                                      //
// Author            Creation Date                                      //
// ======            =============                                      //
// P.A. Lee          12 December 1995                                   //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

#include <iostream>
using namespace std;

#include "error.h"
#include "Text.h"
#include "Account.h"

const int NOT_INITIALIZED = 0;

                         /////////////////////////////
                         // Constructor definitions //
                         /////////////////////////////

Account::Account()
{
  number         = NOT_INITIALIZED;
  currentBalance = 0;
  maxOverdraft   = 0;
  state          = OPEN;
  return;
} // end Account() 

                       /////////////////////////////////
                       // Member Function definitions //
                       /////////////////////////////////

void Account::deposit(int amountMoney)
{
  accountOK();
  currentBalance += amountMoney;
  return;
} // end deposit()

bool Account::withdraw(int howMuch)
{
  accountOK();
  if (howMuch > (currentBalance + maxOverdraft))
  { // If we get here, the amount being requested is too much for the current
    //   balance and overdraft limit. Check state of account
    if (state != SPECIAL)
    { 
      return false;             // Not in SPECIAL state, don't permit
    }
  }  
 currentBalance -= howMuch;     // Decrease the amount in the account
 if (currentBalance < 0)        // Has account become overdrawn?
 {
   if (state != SPECIAL)
   {
     state = OVERDRAWN;         // Set OVERDRAWN if it's not SPECIAL
   }
 }
 return true;
} // end withdraw()

int  Account::balance()
{
  accountOK();
  return currentBalance;
} // end balance()

int  Account::overdraftLimit()
{
  accountOK();
  return maxOverdraft;
} // end overdraftLimit()

int  Account::accountNumber()
{
  accountOK();
  return number;
} // end accountNumber()

State Account::accountState()
{
  accountOK();
  return state;
} // end accountState()

void Account::setOverdraftLimit(int newLimit)
{
  maxOverdraft = newLimit;
  return;
} // end setOverdraftLimit()

void Account::setAccountNumber(int newNumber)
{
  if (isInitialized())
  {
    error("Account::setAccountNumber()"
          " - the account already has a number");
  }
  else
  {
    number = newNumber;
  }
  return;
} // end setAccountNumber()

void Account::setAccountState(State newState)
{
  switch (newState)
  {
    case OPEN:
      {
        if (!isInitialized())
        {
          error("Account:setAccountState(OPEN): account number hasn't been set");
        }
        if (currentBalance < 0)
        {
          state = OVERDRAWN;
        }
        else
        {
          state = OPEN;
        }
      }
      break;
    case STOPPED:
    case SPECIAL:
      {
        state = newState;
      }
      break;
    case OVERDRAWN:
      {
        error("Account::setAccountState(). Setting OVERDRAWN is not permitted");
      }
      break;
    case CLOSED:
      {
        if (currentBalance < 0)
        {
          error("Account::setAccountState(). Account is in deficit; cannot close");
        }
        else if (currentBalance > 0)
        {
          error("Account::setAccountState(). Account is in credit; cannot close");
        } 
        state = CLOSED;      
      }
      break;
    default:
      {
        error("Account:setAccountState() - unknown state specified");
      }
      break;
  } // end switch (newState)
  return;
} // end setAccountState()

                   /////////////////////////////////////////
                   // Private member function definitions //
                   /////////////////////////////////////////

// isInitialized() - return true if initialized, false otherwise

bool Account::isInitialized()
{
  return (number != NOT_INITIALIZED);
} // end isInitialized()

// accountOK() - check the account, give an error and halt
//               if the account shouldnt be manipulated

void Account::accountOK()
{
  if (!isInitialized())
  {
    error("Uninitialized Account variable");
  }
  switch (state)
  {
    case OPEN: case OVERDRAWN: case SPECIAL:
      break;  // Nothing to do for these cases
    case CLOSED:
      {
        error("Account:: This account has been closed");
      }
      break;
    case STOPPED:
      {
        error("Account:: This account has been stopped."
              " Please call the manager");
      }
      break;
    default:
      {
        error("Account::accountOK() - unexpected state");
      }
      break;
  } // end switch
  return;
} // end accountOK()

                         //////////////////////////////
                         // I/O Operator definitions //
                         //////////////////////////////

ostream& operator<<(ostream& outStream, Account value)
{
  outStream << "Account Number:  " << value.accountNumber()  << endl
            << "Balance:         " << value.balance()        << endl
            << "Overdraft Limit: " << value.overdraftLimit() << endl
            << "Account State:   " << value.accountState()   << endl;
  return outStream;
} // end operator<<(Account)

