Windows:

addpath ('c:\matlab\fdaM')
addpath ('c:\matlab\fdaM\examples\weather')

Unix:

addpath('/export/home/steve/ramsay/u1/fdaM')
addpath('/export/home/steve/ramsay/u1/fdaM/examples/weather')

%  Last modified  18 April 2001

%  -----------------------------------------------------------------------
%                      Monthly Weather data
%  -----------------------------------------------------------------------

%  ----------------   input the data and set up labels -------------------

fid = fopen('monthtemp.dat','rt');
tempvec = fscanf(fid,'%f');
tempmat = reshape(tempvec, [12, 35]);

fid = fopen('monthprec.dat','rt');
precvec = fscanf(fid,'%f');
precmat = reshape(precvec, [12, 35]);

meteonames =   [  'St. Johns    '; 'Charlottetown'; 'Halifax      '; ...
                  'Sydney       '; 'Yarmouth     '; 'Fredericton  '; ...
                  'Arvida       '; 'Montreal     '; 'Quebec City  '; ...
                  'Schefferville'; 'Sherbrooke   '; 'Kapuskasing  '; ...
                  'London       '; 'Ottawa       '; 'Thunder Bay  '; ...
                  'Toronto      '; 'Churchill    '; 'The Pas      '; ...
                  'Winnipeg     '; 'Prince Albert'; 'Regina       '; ...
                  'Beaverlodge  '; 'Calgary      '; 'Edmonton     '; ...
                  'Kamloops     '; 'Prince George'; 'Prince Rupert'; ...
                  'Vancouver    '; 'Victoria     '; 'Dawson       '; ...
                  'Whitehorse   '; 'Frobisher Bay'; 'Inuvik       '; ...
                  'Resolute     '; 'Yellowknife  '];

monthletter = ['J'; 'F'; 'M'; 'A'; 'M'; 'J'; 'J'; 'A'; 'S'; 'O'; 'N'; 'D'];

months = ['Jan'; 'Feb'; 'Mar'; 'Apr'; 'May'; 'Jun';
          'Jul'; 'Aug'; 'Sep'; 'Oct'; 'Nov'; 'Dec'];

%  indices for weather stations in each of four climate zones

atlindex = [1,2,3,4,5,6,7,8,9,10,11,13,14,16];
pacindex = [25,26,27,28,29];
conindex = [12,15,17,18,19,20,21,22,23,24,30,31,35];
artindex = [32,33,34];

%  -----------------  set up argument values and weights  ---------------

monthtime = linspace(0.5, 11.5, 12)';  %  mid points of months
weeks     = linspace(0,12,53)';        %  month values roughly in weeks

%  -------------------  set up the fourier basis object -----------------

nbasis = 12;
monthbasis = create_fourier_basis([0,12], nbasis);

%  ------------  set up the harmonic acceleration operator  -------------

%  This operator has the shifted sinusoid as its null space, and it will
%  be used in situations where we want to smooth towards a function
%  that is a combination of a constant, a sine, and a cosine, with period
%  12 months.

Lbasis = create_constant_basis([0,12]);
Lcoef = [0,(pi/6)^2,0];
harmaccelLfd = fd(Lcoef, Lbasis);

%  --------------  make temperature fd object, don't smooth  ------------

tempfd = data2fd(tempmat, monthtime, monthbasis);

tempfd_fdnames{1} = 'Months';
tempfd_fdnames{2} = 'Station';
tempfd_fdnames{3} = 'Deg C';
tempfd = putnames(tempfd, tempfd_fdnames);

%  plot temperature functions

plot(tempfd);
title('Temperature Functions');

%  --------------  make precipitation fd object, don't smooth  ------------

precfd = data2fd(precmat, monthtime, monthbasis);

precfd_fdnames{1} = 'Months';
precfd_fdnames{2} = 'Station';
precfd_fdnames{3} = 'mm';
precfd = putnames(precfd, precfd_fdnames);

%  plot precipitation functions

plot(precfd);
title('Precipitation Functions')

%  ---------------  interactive plots  ---------------------------------

global tempstr precstr nbasis meteonames

tempstr.y = tempmat;
tempstr.c = getcoef(tempfd);
tempstr.mean = mean(tempfd);

precstr.y = precmat;
precstr.c = getcoef(precfd);
precstr.mean = mean(precfd);


monthlymenu


%  ---------------  plot temperature for 4 stations --------------------

stnindex = [8, 24, 27, 34];

plot(tempfd(stnindex))
axis([0,12,-35,25])
title('Selected Temperature Functions')
legend(meteonames(stnindex,:))

%  ----------------------------------------------------------------------
%                Descriptive Statistics Functions
%  ----------------------------------------------------------------------

%  --  compute and plot mean and standard deviation of temperature ------

tempmeanfd   = mean(tempfd);

tempstddevfd = std(tempfd);

subplot(2,1,1);
plot(tempmeanfd);
title('Mean Temperature');

subplot(2,1,2);
plot(tempstddevfd);
title('Std. dev. of Temperature');
axis([0 12 0 10])

%  -----  plot the temperature variance bivariate function  --------

tempvarbifd = var(tempfd);

tempvarmat = eval(tempvarbifd, weeks, weeks);

subplot(1,1,1);
surf(tempvarmat);
xlabel('Weeks')
ylabel('Weeks')
zlabel('Covariance')
title('Temperature Variance-Covariance Function')

%  -------------  plot the correlation function  -----------------------

tempstddev = sqrt(diag(tempvarmat));
tempcormat = tempvarmat./(tempstddev*tempstddev');

subplot(1,1,1);
surf(tempcormat);
xlabel('Weeks')
ylabel('Weeks')
zlabel('Covariance')
title('Temperature Correlation Function')
axis([0,53,0,53,0,1])

%  -----  plot the precipitation variance bivariate function  --------

precvarbifd = var(precfd);

precvarmat = eval(precvarbifd, weeks, weeks);

subplot(1,1,1);
surf(precvarmat);
xlabel('Weeks')
ylabel('Weeks')
zlabel('Covariance')
title('Precipitation Variance-Covariance Function')

%  -------------  plot the correlation function  -----------------------

precstddev = sqrt(diag(precvarmat));
preccormat = precvarmat./(precstddev*precstddev');

subplot(1,1,1);
surf(preccormat);
xlabel('Weeks')
ylabel('Weeks')
zlabel('Covariance')
title('Precipitation Correlation Function')
axis([0,53,0,53,0,1])

%  -----  compute and plot the covariance between temp. and prec.  --

covbifd = var(tempfd, precfd);
covmat  = eval(covbifd,weeks,weeks);

contour(covmat)
xlabel('Months')
ylabel('Months')
title('Covariance')

surf(covmat)
xlabel('Months')
ylabel('Months')
zlabel('Covariance')

%  -----  compute and plot the correlation between temp. and prec.  --

cormat  = covmat./(tempstddev*precstddev');

contour(cormat)
xlabel('Months')
ylabel('Months')
title('Correlation')

surf(cormat)
xlabel('Months')
ylabel('Months')
zlabel('Correlation')

%  ----------------------------------------------------------------------
%               Principal Components Analysis of temperature  
%  ----------------------------------------------------------------------

%  ------------------  center and plot the temperature data  ------------

tempcenterfd = center(tempfd);

plot(tempcenterfd);
title('Centered Temperature');

%  Penalize harmonic acceleration

nharm  = 4;
lambda = 1e-3;
Lfd    = harmaccelLfd;

temppcastr = pca(tempfd, nharm, lambda, Lfd);

%  plot harmonics

subplot(1,1,1)
plot_pca(temppcastr);

%  plot log eigenvalues,
%     passing a line through those from 5 to 12

tempharmeigval = temppcastr.eigvals;
x = ones(8,2);
x(:,2) = reshape((5:12),[8,1]);
y = log10(tempharmeigval(5:12));
c = x\y;
subplot(1,1,1)
plot(1:12,log10(tempharmeigval(1:12)),'-o', 1:12, c(1)+ c(2).*(1:12), ':')
xlabel('Eigenvalue Number')
ylabel('Log10 Eigenvalue')

%  plot pca scores

tempharmscr = temppcastr.harmscr;
plot(tempharmscr(:,1),tempharmscr(:,2), 'o')
xlabel('Scores on Harmonic 1')
ylabel('Scores on Harmonic 2')
text(tempharmscr(:,1),tempharmscr(:,2),meteonames)

%  ----------------------------------------------------------------------
%          Linear model for temperature using climate zones  
%          This is a functional one-way analysis of variance
%  ----------------------------------------------------------------------

%  setup design matrix.  Make it full rank and use atlantic zone
%     as baseline group

zmat = zeros(35,4);
zmat(:       ,1) = 1;
zmat(pacindex,2) = 1;
zmat(conindex,3) = 1;
zmat(artindex,4) = 1;

%  estimate linear model

linmodstr = linmod(zmat, tempfd);

%  plot four regression functions

tempregfd = linmodstr.reg;

plot(tempregfd)
title('Regression Functions')

%  plot approximation functions (there are only 4 distinct ones)

tempyhatfd = linmodstr.yhat;

plot(tempyhatfd)
title('Model Functions')

%  compute residual functions

tempresfd = tempfd - tempyhatfd;

%  compute mean of temperatures

tempmeanfd = mean(tempfd);

%  Compute the error sum of squares function

tempresmat = eval(tempresfd, monthtime);
SSE = sum((tempresmat').^2)';

plot(monthtime, SSE)
xlabel('Month')
ylabel('SSE')

%  Compute error sum of squares about mean

tempresmat0 = eval(tempfd, monthtime) - ...
              eval(tempmeanfd, monthtime)*ones(1,35);
SSY = sum((tempresmat0').^2)';

%  Compute squared multiple correlaton and F-ratio functions

RSQ = (SSY - SSE)./SSY;
Fratio = ((SSY - SSE)./3)./(SSE./31);

%  Plot these functions, 
%    along with 0.05 critical value for F for 3 and 31 df

subplot(1,2,1)
plot(monthtime, RSQ)
xlabel('Month')
title('R^2')
axis([0,12,0,1])
axis('square')
subplot(1,2,2)
plot(monthtime, Fratio, '-', [0,12], [2.9,2.9], '--')
xlabel('Month')
title('F')
axis([0,12,0,40])
axis('square')

%  Set up a design matrix having a column for the grand mean, and
%    a column for each climate zone effect. Add a dummy contraint
%    observation

zmat = zeros(35,5);
zmat(:       ,1) = 1;
zmat(atlindex,2) = 1;
zmat(pacindex,3) = 1;
zmat(conindex,4) = 1;
zmat(artindex,5) = 1;
%  attach a row of 0, 1, 1, 1, 1
z36    = ones(1,5);
z36(1) = 0;
zmat = [zmat; z36];

%  set up a new fd by adding a zero function

coef = getcoef(tempfd);
coef36 = [coef,zeros(13,1)];
tempfd36 = putcoef(tempfd, coef36);

%  fit linear model

linmodstr = linmod(zmat, tempfd36);

% plot five regression functions

tempregfd = linmodstr.reg;

plot(tempregfd);
title('Regression Functions')

%  plot residual functions

tempyhatfd = linmodstr.yhat;

tempresfd = tempfd36 - tempyhatfd;

plot(tempresfd)
title('Residuals')

%  ----------------------------------------------------------------------
%       now model log mean precip. as a function of temperature  
%  ----------------------------------------------------------------------

%  Compute log precipitation

logannprec = log10(sum(precmat)');
logannprec = logannprec - mean(logannprec);

xLfd = harmaccelLfd;
xlambda = 1e-2;

yLfd = 0;      %  not actually used for this case
ylambda = 0;   %  not actually used for this case

wtvec     = ones(35,1);  %  weight vector for observations used in linmod

linmodstr = linmod(tempfd, logannprec, wtvec, ...
                   xLfd, yLfd, xlambda, ylambda);

intercept    = linmodstr.alpha;
regressionfd = linmodstr.reg;

%  plot regression function

plot(regressionfd);
title('Regression of Log Annual Precipitation on Temperature');

%  compute fitted values and plot Y against Yhat

yhat = inprod(regressionfd,tempfd) + intercept;
plot(yhat, logannprec, 'o', yhat, yhat, '--')
xlabel('Predicted log annual precipitation')
ylabel('Log annual precipitation')

%  compute squared correlation

covmat = cov(yhat,logannprec);
RSQ    = covmat(1,2)^2/(covmat(1,1)*covmat(2,2))

%  -------------------------------------------------------------------
%              Predict log precipitation from temperature   
%   The regression coefficient function is now bivariate:  \beta(s,t)
%  -------------------------------------------------------------------

%  set up functional data object for log precipitation

lnprecfd = data2fd(log10(precmat), monthtime, monthbasis);

lnprecfd_fdnames{1} = 'Months';
lnprecfd_fdnames{2} = 'Station';
lnprecfd_fdnames{3} = 'log_{10} mm';
lnprecfd = putnames(lnprecfd, lnprecfd_fdnames);

%  plot precipitation functions

plot(lnprecfd);
title('Log Precipitation Functions')

%  set up smoothing levels for s (xLfd) and for t (yLfd)

xLfd = harmaccelLfd;
yLfd = harmaccelLfd;
xlambda = 1e-2;
ylambda = 0;

%  compute the linear model

linmodstr = linmod(tempfd, lnprecfd, wtvec, ...
                   xLfd, yLfd, xlambda, ylambda);

afd = linmodstr.alpha;   %  The intercept function
bfd = linmodstr.reg;     %  The bivariate regression function

%  plot the intercept function

plot(afd);

%  plot the regression function

bfdmat = eval(bfd, weeks, weeks);

image(bfdmat)
xlabel('Month (t)')
ylabel('Month (s)')
title('Regression Function \beta(s,t)')
axis([0,12,0,12,-.25,.25])

surf(weeks, weeks, bfdmat)
xlabel('Month (t)')
ylabel('Month (s)')
title('Regression Function \beta(s,t)')
axis([0,12,0,12,-.25,.25])

%  plot the regression function as a function of s 
%    for selected values of t.  Press any key to advance plot

for t=1:2:53
    plot(weeks, bfdmat(:,t), '-', [0,12], [0,0], '--')
    xlabel('Month (s)')
    ylabel('\beta(s,t)')
    title(['t = ',num2str(weeks(t))])
    axis([0,12,-.25,.25])
    pause
end

%  Get fitted functions

lnprechatfd = linmodstr.yhat;

% Compute mean function as a benchmark for comparison

lnprecmeanfd = mean(lnprecfd);

%  Plot actual observed, fitted, and mean log precipitation for
%      each weather station, 

lnprechat0 = eval(lnprecmeanfd,weeks);
for i=1:35
    lnpreci    = eval(lnprecfd(i),    weeks);
    lnprechati = eval(lnprechatfd(i), weeks);
    SSE = sum((lnpreci-lnprechati).^2);
    SSY = sum((lnpreci-lnprechat0).^2);
    RSQ = (SSY-SSE)/SSY;
    plot(weeks, lnpreci, 'o', weeks, lnprechati, '-', ...
                              weeks, lnprechat0, '--')
    xlabel('Month')
    ylabel('Log Precipitation')
    title([meteonames(i,:),'  R^2 = ',num2str(RSQ)])
    axis([0,12,0.6,2.6])
    pause
end

%  ---------------------------------------------------------------
%                   Register temperature data   
%  ---------------------------------------------------------------

%  register the first derivative of temperature

%  first smooth the temperature functions 

stempfd = smooth(tempfd, 1e-3, harmaccelLfd);

%  set up the basis for the warping function

nbasis = 5;
wbasis = create_fourier_basis([0,12],nbasis);

%  set up parameters for the registration function

Lfd      = 3;
lambda   = 1e-3;

index = 1:35;
Dtempfd = deriv(stempfd(index),1);
y0fd = mean(Dtempfd);
index = [1, 8, 24, 27, 32];  %  register a subset of the stations
yfd  = Dtempfd(index);
xfine = linspace(0,12,101)';
ofine = ones(101,1);
y0vec = eval(y0fd, xfine);
yvec  = eval(yfd, xfine);

cvec0  = zeros(nbasis,length(index));
Wfd0   = fd(cvec0, wbasis);

regstr = registerfd(y0fd, yfd, Wfd0, Lfd, lambda, periodic);

yregfd  = regstr.regfd;
yregmat = eval(yregfd,xfine);
Wfd     = regstr.Wfd;
shift   = regstr.shift;
warpmat = monfn(xfine, Wfd);
warpmat = ofine*shift' + 12.*warpmat./(ofine*warpmat(101,:));

for i = 1:length(index)
   subplot(1,2,1)
   plot(xfine, yvec(:,i), '-', xfine, y0vec, '--', xfine, yregmat(:,i), '-');
   axis('square')
   title(meteonames(index(i),:))
   subplot(1,2,2)
   plot(xfine, warpmat(:,i), '-', xfine, xfine+shift(i), '--')
   axis('square')
   title(['Shift = ',num2str(shift(i))])
   pause
end

%  -----------------------------------------------------------------------
%      Smooth the temperature data using harmonic acceleration
%        The standard error is about 0.2 dec C.
%  -----------------------------------------------------------------------

eyefd = data2fd(eye(12), monthtime, monthbasis);

lambda = 1.2e-4;   %  correcting for df of fit
lambda = 2.5e-3;   %  uncorrected

tempsmthfd = smooth(tempfd, lambda, harmaccelLfd);

smthopr = smooth(eyefd, lambda, harmaccelLfd);
smthmat = eval(smthopr, monthtime);
df = sum(diag(smthmat))

tempsmthmat = eval(tempsmthfd, monthtime);

resmat = tempmat - tempsmthmat;

varest = sum(sum(resmat.^2))./(35*(12-df));
varest = mean(mean(resmat.^2));
stderr = sqrt(varest)

%  plot fit and residuals

for i=1:35
    subplot(1,2,1)
    plot(monthtime, tempmat(:,i), 'o', monthtime, tempsmthmat(:,i), '-')
    axis([0,12,-35,25])
    axis('square')
    title(meteonames(i,:))
    subplot(1,2,2)
    plot(monthtime, resmat(:,i), 'o-', [0,12], [0,0], '--')
    axis([0,12,-.7,.7])
    axis('square')
    pause
end

%  --------------------------------------------------------------------
%             Principal Differential Analysis
%  --------------------------------------------------------------------

difeorder = 3;
estimate = [0, 1, 1]';
lambda   = 1000000.*ones(3,1);
lambda   = zeros(3,1);

npdabasis = 11;
wbasis  = create_fourier_basis([0,12], npdabasis);
wcoef   = zeros(npdabasis,difeorder);
wcoef(1,:) = [0,(pi/6)^2/sqrt(1/12),0];
wfd0 = fd(wcoef,wbasis);

wfd = pda(tempsmthfd, difeorder, wbasis, lambda, wfd0, estimate);

subplot(1,1,1)
plot(wfd)

wmat = eval(wfd,weeks);

plot(weeks, wmat(:,1), 'k-.', weeks, wmat(:,2), 'k-', weeks, wmat(:,3), 'k--', ...
     [0,12], [(pi/6)^2, (pi/6)^2], 'k:')
xlabel('\fontsize{16} Month')
ylabel('\fontsize{16} Weight functions w_j(t)')
legend('\fontsize{12} w_0(t)', 'w_1(t)', 'w_2(t)', '(\pi/6)^2')

print -dps2 'wfns.ps'

%  plot forcing functions

forcemat1 = eval(tempsmthfd, monthtime, wfd);
forcemat0 = eval(tempsmthfd, monthtime, harmaccelLfd);

RMSE0 = sqrt(mean(mean(forcemat0.^2)))
RMSE1 = sqrt(mean(mean(forcemat1.^2)))

for i=1:35
    force0i = forcemat0(:,i);
    force1i = forcemat1(:,i);
    rmse0   = sqrt(mean(force0i.^2));
    rmse1   = sqrt(mean(force1i.^2));
    plot(monthtime, forcemat1(:,i), 'o-', monthtime, forcemat0(:,i), 'o-', ...
         monthtime, tempsmthmat(:,i), '--', [0,12], [0,0], ':')
    axis([0,12,-35,25])
    axis('square')
    title([meteonames(i,:),' RMSE0 = ',num2str(rmse0),' RMSE1 = ',num2str(rmse1)])
    pause
end

%  solve equation

global wfd  %  this is necessary for function derivs

ystart = eye(3);
[tp1, yp1] = ode45('derivs', monthtime, ystart(:,1));
[tp2, yp2] = ode45('derivs', monthtime, ystart(:,2));
[tp3, yp3] = ode45('derivs', monthtime, ystart(:,3));

%  plot the three solutions

umat = [yp1(:,1),yp2(:,1),yp3(:,1)];

[tp1, yp1] = ode45('derivs', weeks, ystart(:,1));
[tp2, yp2] = ode45('derivs', weeks, ystart(:,2));
[tp3, yp3] = ode45('derivs', weeks, ystart(:,3));

umatw = [yp1(:,1),yp2(:,1),yp3(:,1)];

subplot(2,1,1)
plot(monthtime, umat, monthtime, zeros(51,1), ':'), title('Function');
Dumat = [yp1(:,2),yp2(:,2)];
subplot(2,1,2)
plot(monthtime, Dumat, monthtime, zeros(51,1), ':'), title('Derivative');

vmat  = [ones(12,1), sin(pi.*monthtime./6), cos(pi.*monthtime./6)];
vmatw = [ones(53,1), sin(pi.*weeks./6), cos(pi.*weeks./6)];

i = 24;

y = tempmat(:,i);
yhatv = vmatw * (vmat\y);
yu = eval(tempfd(i), weeks);
yhatu = umatw * (umatw\yu);

plot(monthtime, y, '.', weeks, yhatv, '-', weeks, yhatu, '--')

