function s = scalar_product(coef1, coef2, k1, k2, breaks, deriv)
%SCALAR_PRODUCT computes the scalar product between two functions f and g  
%    expressed as linear combinations of B-splines
%
% f = coef1(1)*B(1,k1,t1) + ... + coef1(L-1+k1)*B(L-1+k1,k1,t1),
% g = coef2(1)*B(1,k2,t2) + ... + coef2(L-1+k2)*B(L-1+k2,k2,t2), 
% L = number of breaks - 1,
% t = [breaks(1)*ones(1,k1), 
%      breaks(2),breaks(3),...,breaks(L),
%      breaks(L+1)*ones(1,k1)]  
% B(i,k,t) = B(i,k)(x) 
%          = (t(i+k)-t(i))*[t(i),t(i+1),...,t(i+k)]*(.-x)^{k-1}_{+}.
% We assume that BREAKS is a strictly increasing sequence. 
% The function returns the integrals of the product f*g
%  over the interval [breaks(1),breaks(L+1)].
%
%  Arguments:
%  coef1:   Coefficient vector of length # BREAKS - 2 + K1
%  coef2:   Coefficient vector of length # BREAKS - 2 + K2
%  K1:      Order of B-splines for first  argument
%  K2:      Order of B-splines for second argument
%  BREAKS:  Break point sequence common to both arguments.
%           It must be strictly increasing.
%  DERIV:   Order of derivative

%  Last modified 20 February 2002

if nargin < 6, deriv = 0; end

%  check breaks sequence

if length(breaks) < 2
    error('The length of argument BREAKS is less than 2.');
end

breakdiff = diff(breaks);
if min(breakdiff) <= 0
    error('Argument BREAKS is not strictly increasing.');
end

n  = size(breaks,2);
L  = n - 1;       %  number of intervals defined by BREAKS
M1 = L + k1 - 1;  %  number of basis functions for first  spline
M2 = L + k2 - 1;  %  number of basis functions for second spline

if size(coef1) ~= M1 | size(coef2) ~= M2
    error(['Error: coef1 should be should have length #breaks+k1-2',
           ' and coef2 #breaks+k2-2.']);
end

breaks1 = breaks(1);
breaksn = breaks(n);

% The knot sequences are built so that there are no continuity conditions 
% at the first and last breaks.  There are k-1 continuity conditions at 
% the other breaks.
temp = breaks(2:(n-1));
t1   = [breaks1*ones(1,k1),temp,breaksn*ones(1,k1)]; 
t2   = [breaks1*ones(1,k2),temp,breaksn*ones(1,k2)]; 

% Construct  the piecewise polynomial representation of 
%    f^(DERIV) and g^(DERIV)

% Initialize the piecewise polynomial representation of f^(DERIV)

polycoef1 = zeros(L,k1-deriv); 
for i = 1:M1 
    [Coeff,index] = ppBspline(t1(i:i+k1));
    % convert the index of the breaks in t1 to the index in the
    % variable 'breaks'
    index = index + i - k1;
    CoeffD = ppderiv(Coeff,deriv); % differentiate B(i,k1,t1)(x)
    % add the polynomial representation of B(i,k1,t1)(x) to f
    polycoef1(index,:) = coef1(i).*CoeffD + polycoef1(index,:); 
end

polycoef2 = zeros(L,k2-deriv); 
for i = 1:M2 
    [Coeff,index] = ppBspline(t2(i:i+k2));
    % convert the index of the breaks in t2 to the index in the 
                            % variable 'breaks'
    index = index + i - k2; 
    CoeffD = ppderiv(Coeff, deriv); % differentiate B(i,k2,t2)(x)
    % add the polynomial representation of B(i,k2,t2)(x) to g
    polycoef2(index,:) = coef2(i).*CoeffD + polycoef2(index,:); 
end

% Compute the scalar product between f and g

s = 0; % initialize the scalar product
for j = 1:L
    % multiply f and g piecewise and integrate
    polyprod = conv(polycoef1(j,:),polycoef2(j,:));
    Pint = polyint(polyprod); 
    sj   = polyvalm(Pint,breaks(j+1) - breaks(j  )); % add the integral to s
    s = s + sj;
end
