/************************************************************************
 * CASSI, version 1.01
 * Copyright 2012,
 * Richard Howey
 * Institute of Genetic Medicine, Newcastle University
 *
 * richard.howey@ncl.ac.uk
 * http://www.staff.ncl.ac.uk/richard.howey/
 *
 * This file is part of CASSI, the SNP interaction analysis program.
 *
 * CASSI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * CASSI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CASSI.  If not, see <http://www.gnu.org/licenses/>.
 ************************************************************************/


/*! \file Analysis.cpp
    \brief This file contains the methods the various analyse.
    
*/
#include <iostream>
#include <sstream>

using namespace std; // initiates the "std" or "standard" namespace

#include "Analysis.h"
#include "Statistics.h"
#include "main.h"

//! Runs the chosen analysis.
void Analysis::runAnalysis()
{
	//setup SNP description info. for both files
	DescriptionOfSNPs * descSNPs = new DescriptionOfSNPs(filename, filename2);

	if(snp1EndSNP == 0) snp1EndSNP = descSNPs->getNoSNPs(false);

	if(snp2EndSNP == 0)
	{
			if(filename2 == "") snp2EndSNP = descSNPs->getNoSNPs(false);
			else snp2EndSNP = descSNPs->getNoSNPs(true);
	};

	//create statistic to test
	JointEffects jointEffects(filename, filename2, outputFileName, thresholdCC, thresholdCO, maxNoResults, descSNPs);

	bool differentData = true;

	//if the second filename is not given use the first one again, set now to avoid unnec. checks between files for pedigree data 
	if(filename2 == "")
	{
		filename2 = filename;
		differentData = false;
	};

	//setup the SNP windows to analyse, advance thro' data file to before first SNPs in windows
	SNPWindow * snpWindow1 = new SNPWindowReadFromFile(jointEffects.getTotalNoSubjects(), snp1StartSNP, filename, true);
	SNPWindow * snpWindow2 = new SNPWindowStoreAllData(jointEffects.getTotalNoSubjects(), snp2StartSNP, filename2, snp2EndSNP);

	//add a ref to the SNP data created in windows to the statistic
	jointEffects.setSNPWindows(snpWindow1, snpWindow2);

	unsigned int snp1No = snp1StartSNP;
	unsigned int snp2No = snp2StartSNP;

	
	//loop through SNP 1
	for( ; ; )
	{

		//start reading the data from the start of SNP window 2
		snpWindow2->startWindowAtStart();

		snp2No = snp2StartSNP;

		//loop through SNP 2 to analysis SNP 1 against
		for( ; ; )
		{			
			
			//Analysis SNP 1 against SNP 2, but do not repeat analysis or analyse against the same SNP
			if(differentData || snp2No > snp1No || snp2No < snp1StartSNP || snp1No > snp2EndSNP)
			{
		
				//calculate the joint genotype counts for SNP1 and SNP2
				jointEffects.updateJointGenotypeCounts();

				//calculate the statistic and keep result if below threshold
				jointEffects.evaluateStatistic(snp1No, snp2No);
							
			};

			//move to the next SNP
			++snp2No;

			if(snp2No > snp2EndSNP) break;

			//read in new SNP data for SNP 2
			snpWindow2->moveToNextSNP();
		};

		//move to the next SNP
		++snp1No;

		if(snp1No > snp1EndSNP) break;

		//read in new SNP data for SNP 1
		snpWindow1->moveToNextSNP();
	};

	jointEffects.outputTotalResults();

	delete snpWindow1;
	delete snpWindow2;
	delete descSNPs;
};

//! Converts an integer to a string
string toString(int & i)
{
	ostringstream aStringStream;
	aStringStream << i;

	return aStringStream.str();
};

//! Returns a string of the run time
string getTime(const double & t)
{
	double time = t;
	int days = 0;
	int hours = 0;
	int minutes = 0;
	int seconds = 0;

	string ans = "";
	days = (int) (time / 86400); time -= days*86400;
	hours = (int) (time / 3600); time -= hours*3600;
	minutes = (int) (time / 60); time -= minutes*60;
	seconds = (int) time;

	if(days == 1) ans += "1 day";
	else if(days > 0) ans += toString(days) + " days";

	if(hours > 0)
	{
		if(days != 0)
		{
			if(minutes == 0 && seconds == 0) ans += " and ";
			else ans += ", ";
		};

		if(hours == 1) ans += "1 hour";
		else ans += toString(hours) + " hours";
	};

	if(minutes > 0)
	{
		if(ans != "")
		{
			if(seconds == 0) ans += " and ";
			else ans += ", ";
		};

		if(minutes == 1) ans += "1 minute";
		else ans += toString(minutes) + " minutes";
	};

	if(seconds > 0)
	{
		if(ans != "")
		{
			ans += " and ";			
		};

		if(seconds == 1) ans += "1 second";
		else ans += toString(seconds) + " seconds";
	};

	if(ans == "") ans = "less than one second";

	return ans;
};





