/************************************************************************
 * CASSI, version 1.01
 * Copyright 2012,
 * Richard Howey
 * Institute of Genetic Medicine, Newcastle University
 *
 * richard.howey@ncl.ac.uk
 * http://www.staff.ncl.ac.uk/richard.howey/
 *
 * This file is part of CASSI, the SNP interaction analysis program.
 *
 * CASSI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * CASSI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CASSI.  If not, see <http://www.gnu.org/licenses/>.
 ************************************************************************/


/*! \file Data.h
    \brief This file contains classes for manipulating SNP data.
    
*/

#ifndef __DATA
#define __DATA

#include <set>
#include <map>
#include <list>
#include <stack>
#include <fstream>

using namespace std;


//! Stores SNP data for all subjects for a given SNP.
struct SNPData
{
	
	list<unsigned int> noMinorAllelesAllSubjects; //ordered list of the number of minor alleles for the subjects	
	
	SNPData() : noMinorAllelesAllSubjects() {};
	
	~SNPData()
	{


	};

	
};

//! Joint Genotype Counts
struct JointGenotypeCounts
{
	unsigned int total;
	unsigned int counts[3][3];

	double adjCount00, adjCount01, adjCount02, adjCount10, adjCount11, adjCount12, adjCount20, adjCount21, adjCount22; 
	bool halfAdded;

	JointGenotypeCounts()
	{
		resetCounts();	
	};

	void addCount(const unsigned int & count1, const unsigned int & count2);
	void setAdjCounts(const bool & addHalf);
	void resetCounts();
};

void setCounts(JointGenotypeCounts * jgc1, JointGenotypeCounts & jgc2);

//! General class for a SNP window
class SNPWindow
{
private:

	unsigned int totalNoSubjects;
	unsigned int startSNPNo;
	string filename;
	unsigned int bitCount;
	int one;
	int aBit;	
	char buffer[1];
	ifstream readSNPData;

public:

	SNPWindow(const unsigned int & ts, unsigned int & ssn, string & fname) : totalNoSubjects(ts), startSNPNo(ssn), filename(fname), bitCount(9), one('\1') {};
	

	virtual ~SNPWindow()
	{		
		if(readSNPData.is_open()) readSNPData.close();	
	};

	void advanceSNPData();
	void advanceToFirstWindow();
	unsigned int getNextNoOfMinorAlleles();
	void startNewByte() {bitCount = 9;};
	void reopenBinaryFile();
	void openBinaryFileFirst();

	virtual void moveToNextSNP() {};
	virtual void startWindowAtStart() {};
	virtual SNPData * getCurrentSNP() {return 0;};
};

//! Stores SNP data for the SNP window in question.
class SNPWindowReadFromFile : public SNPWindow
{
private:

	SNPData * snp; //this SNP object is also referenced in the SNP statistic

public:

	SNPWindowReadFromFile(const unsigned int & ts, unsigned int & startSNPNo, string & fname, const bool & readFirstSNP);
	

	~SNPWindowReadFromFile()
	{		
		delete snp;		
	};

	void moveToNextSNP();
	void startWindowAtStart();
	SNPData * getCurrentSNP() {return snp;};
};

//! SNP window that stores all the data
class SNPWindowStoreAllData : public SNPWindow
{
private:

	list<SNPData *> allSNPData;
	list<SNPData *>::const_iterator currentSNP;

public:

	SNPWindowStoreAllData(const unsigned int & ts, unsigned int & startSNPNo, string & fname, unsigned int & endSNPNo);
	
	~SNPWindowStoreAllData()
	{		
		for(list<SNPData *>::iterator s = allSNPData.begin(); s != allSNPData.end(); ++s)
		{
			delete *s;
		};
	};

	void moveToNextSNP() {currentSNP++;};
	void startWindowAtStart() {currentSNP = allSNPData.begin();};
	SNPData * getCurrentSNP() {return *currentSNP;};
};

//! Names of SNPs
class DescriptionOfSNPs
{
private:
	map<unsigned int, pair<unsigned int, string> > basePairs1; //base-pair positions of all SNPs in order with SNP names
	map<unsigned int, pair<unsigned int, string> > basePairs2;
	bool oneFile;

public:

	DescriptionOfSNPs(string & filename, string & filename2)
	{
		setUpSNPDesciptionData(filename2, true);
		setUpSNPDesciptionData(filename, false);	
	};

	//methods regarding the SNP descriptions
	void setUpSNPDesciptionData(string & filename, const bool & secondFile);
	unsigned int getNoSNPs(const bool & second) const
	{
		if(second) return basePairs2.size();
		else return basePairs1.size();
	};
	string getSNPName(unsigned int & snpNo, const unsigned int & winNo) const;
};


#endif
