/************************************************************************
 * CASSI, version 1.01
 * Copyright 2012,
 * Richard Howey
 * Institute of Genetic Medicine, Newcastle University
 *
 * richard.howey@ncl.ac.uk
 * http://www.staff.ncl.ac.uk/richard.howey/
 *
 * This file is part of CASSI, the SNP interaction analysis program.
 *
 * CASSI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * CASSI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CASSI.  If not, see <http://www.gnu.org/licenses/>.
 ************************************************************************/


/*! \file Statistics.cpp
    \brief This file contains the source of models used for logistic regression.
    
*/

#include <map>
#include <iostream>
#include <math.h>

#include <string>

using namespace std; // initiates the "std" or "standard" namespace

#include "main.h"
#include "Statistics.h"
#include "Data.h"

const double Statistic::oneOverSqRoot2 = 0.7071067811865475244008443621048490392848359376884740365883;

//! Sets up family case/control data from the .fam file
void Statistic::setupCaseControls(string & filename)
{
	//try and find the family file and read in data
	unsigned int length = filename.length();
	string famFilename = filename.substr(0, length-4) + ".fam";

	ifstream readFamilyFile;
	readFamilyFile.open(famFilename.c_str());
	if(!readFamilyFile.is_open())
	{
		outErr("Cannot read family file: "); outErr(famFilename); outErr("!\n");
		exit(0);
	};

	string famID, indivID, FatherId, MotherID, sexID, famIndivID;
	string prevFamIndivID = "";
	int phenoType;
	unsigned int noCases = 0;
	
	//loop thro' subjects and store the cases
	do{		
		readFamilyFile >> famID >> indivID >> FatherId >> MotherID >> sexID >> phenoType;
		famIndivID = famID + "-" + indivID;

		//do not duplicate the last row
		if(famIndivID != prevFamIndivID) 
		{
			if(phenoType == 2)
			{
					caseControls.push_back(true);					
					noCases++;
			}			
			else caseControls.push_back(false);
		};

		prevFamIndivID = famIndivID;
	}while(!readFamilyFile.eof());

	
	double casesPercent = (((double)(noCases))/((double)(caseControls.size())))*100;
	double controlsPercent = (((double)(caseControls.size() - noCases))/((double)(caseControls.size())))*100;

	out("Number of subjects: "); out(caseControls.size() ); out("\n");
	out("Number of cases: "); out(noCases ); out(" ("); out(casesPercent); out("%)\n");
	out("Number of controls: "); out(caseControls.size() - noCases); out(" ("); out(controlsPercent); out("%)\n");
	out("\n");

	readFamilyFile.close();
};

//! Calculates the p-value from a Chi square value with 1 df.
double Statistic::getPvalueChiSq1DF(double & chisq) const
{
	return 1 - erf(sqrt(chisq)*oneOverSqRoot2);
};

//! Checks the family case/control data from the .fam file against that from first filename
void Statistic::checkCaseControls(string & filename2, string & filename)
{
	if(filename2 == "") return;

	//try and find the family file and read in data
	unsigned int length = filename2.length();
	string famFilename2 = filename2.substr(0, length-4) + ".fam";

	ifstream readFamilyFile;
	readFamilyFile.open(famFilename2.c_str());
	if(!readFamilyFile.is_open())
	{
		outErr("Cannot read family file: "); outErr(famFilename2); outErr("!\n");
		exit(0);
	};

	string famID, indivID, FatherId, MotherID, sexID, famIndivID;
	string prevFamIndivID = "";
	int phenoType;
	unsigned int noCases = 0;
	list<bool>::const_iterator cc = caseControls.begin();

	//loop thro' subjects and store the cases
	do{		
		readFamilyFile >> famID >> indivID >> FatherId >> MotherID >> sexID >> phenoType;
		famIndivID = famID + "-" + indivID;

		//do not duplicate the last row
		if(famIndivID != prevFamIndivID) 
		{
				
				if((!(*cc) && phenoType == 2) || ((*cc) && phenoType != 2))
				{
					outErr("\nCase/Control status for "); outErr(filename); outErr(" and "); outErr(filename2); outErr(" do not match!\n (At subject no. "); outErr(noCases); outErr(")\n");
					exit(0);
				};
					
				cc++;
				noCases++;			
		};

		if(cc == caseControls.end()) break;

		prevFamIndivID = famIndivID;
	}while(!readFamilyFile.eof());

	if(noCases != caseControls.size())
	{
		outErr("\nThe number of subjects in "); outErr(filename); outErr(" and "); outErr(filename2); outErr(" do not match!\n ("); outErr(caseControls.size()); outErr(" and "); outErr(noCases); outErr(" respectively)\n");
		exit(0);
	};

	readFamilyFile.close();
};

//! For testing purposes
void outputMatrix(list< list<double> > & matrix)
{
	out("\n");

	for(list< list<double> >::const_iterator r = matrix.begin(); r != matrix.end(); ++r)
	{
		for(list<double>::const_iterator c = r->begin(); c != r->end(); ++c)
		{
			out(*c); out("\t");
		};
			out("\n");
	};

};

void Statistic::updateJointGenotypeCounts()
{
	//reset all joint genotype counts to zero in order to calculate new counts for next SNP pair
	caseJointGenotypeCounts.resetCounts();
	controlJointGenotypeCounts.resetCounts();

	SNPData * snp1 = window1->getCurrentSNP();

	//loop thro' all the subjects for each SNP
	list<unsigned int>::const_iterator s1 = snp1->noMinorAllelesAllSubjects.begin();
	list<unsigned int>::const_iterator s2 = window2->getCurrentSNP()->noMinorAllelesAllSubjects.begin();
	list<bool>::const_iterator caco = caseControls.begin();

	//count the joint minor alleles for each subject
	do{

		//count cases and controls separately
		if(*caco) caseJointGenotypeCounts.addCount(*s1, *s2);
		else controlJointGenotypeCounts.addCount(*s1, *s2);

		++s1; ++s2; ++caco;
	}while(s1 != snp1->noMinorAllelesAllSubjects.end());

};

//! Calculate the odds ratio relative risk.
double JointEffects::getOddRatioRelRisk(double & num1, double & num2, double & denom1, double & denom2)
{
	return (num1*num2)/(denom1*denom2);
};

//! Return the log value from the cache, if not add it
double JointEffects::getLog(double v)
{
	map<double, double>::const_iterator i = logCache.find(v);
	if(i == logCache.end())
	{
		double logv = log(v);
		logCache[v] = logv;
		return logv;
	}
	else
	{		
		return i->second;
	};
};

void setInv(double & val, unsigned int & count)
{
	if(count == 0) val = 2.0;// set count to 0.5 and take reciprical, giving 2;
	else val = 1.0/(double)(count);
};

//!For testing
void outputCounts(JointGenotypeCounts & jc)
{
	out("\n");
	out(jc.counts[0][0]); out(" "); out(jc.counts[0][1]); out(" "); out(jc.counts[0][2]); out("\n");
	out(jc.counts[1][0]); out(" "); out(jc.counts[1][1]); out(" "); out(jc.counts[1][2]); out("\n");
	out(jc.counts[2][0]); out(" "); out(jc.counts[2][1]); out(" "); out(jc.counts[2][2]); out("\n\n");
};

void JointEffects::calculateis(double & i22, double & i21, double & i12, double & i11, JointGenotypeCounts & jc)
{
	i22 = (jc.adjCount22*jc.adjCount00)/(jc.adjCount20*jc.adjCount02);
	i21 = (jc.adjCount21*jc.adjCount00)/(jc.adjCount20*jc.adjCount01);
	i12 = (jc.adjCount12*jc.adjCount00)/(jc.adjCount10*jc.adjCount02);
	i11 = (jc.adjCount11*jc.adjCount00)/(jc.adjCount10*jc.adjCount01);
};

//! Calculates lambda and the varince needed for the joint effect statistic.
void JointEffects::calculateVariables(double & i22, double & i21, double & i12, double & i11, double & totalInverseV, double rowTotals[], JointGenotypeCounts & jc, const bool & useAltStat)
{
	
	//partial diff of f with xi, a diagonal 4 x 4 matrix
	double pdfpdXi1, pdfpdXi2, pdfpdXi3, pdfpdXi4;

	if(useAltStat)
	{
		pdfpdXi1 = sqrt(i22)/(2*sqrt(i22) + 2);
		pdfpdXi2 = i21/(i21 + 1);
		pdfpdXi3 = i12/(i12 + 1);
		pdfpdXi4 = 1.0;
	}
	else
	{
		pdfpdXi1 = 0.5;
		pdfpdXi2 = 1.0;
		pdfpdXi3 = 1.0;
		pdfpdXi4 = 2*i11/(2*i11 - 1);
	};

	double invq22, invq02, invq20, invq00, invq21, invq12, invq01, invq10, invq11;
	
	invq00 = 1.0/jc.adjCount00;
	invq01 = 1.0/jc.adjCount01;
	invq02 = 1.0/jc.adjCount02;
	invq10 = 1.0/jc.adjCount10;
	invq11 = 1.0/jc.adjCount11;
	invq12 = 1.0/jc.adjCount12;
	invq20 = 1.0/jc.adjCount20;
	invq21 = 1.0/jc.adjCount21;
	invq22 = 1.0/jc.adjCount22;

	//Do not add in n^(-1) scalar factor of matrix C (and V) add into w estimates later 
	
	//build matrix V firstly
	list< list<double> > matrixV, inverse;

	list<double> row1;
	row1.push_back((invq22 + invq02 + invq20 + invq00)*pdfpdXi1*pdfpdXi1);
	double v12 = (invq20 + invq00)*pdfpdXi1*pdfpdXi2;
	row1.push_back(v12);
	double v13 = (invq02 + invq00)*pdfpdXi1*pdfpdXi3;
	row1.push_back(v13);
	double v14 = invq00*pdfpdXi1*pdfpdXi4;
	row1.push_back(v14);

	matrixV.push_back(row1);

	list<double> row2;
	row2.push_back(v12);
	row2.push_back((invq21 + invq20 + invq01 + invq00)*pdfpdXi2*pdfpdXi2);
	double v23 = invq00*pdfpdXi2*pdfpdXi3;
	row2.push_back(v23);
	double v24 = (invq01 + invq00)*pdfpdXi2*pdfpdXi4;
	row2.push_back(v24);

	matrixV.push_back(row2);

	list<double> row3;
	row3.push_back(v13);
	row3.push_back(v23);
	row3.push_back((invq12 + invq10 + invq02 + invq00)*pdfpdXi3*pdfpdXi3);
	double v34 = (invq10 + invq00)*pdfpdXi3*pdfpdXi4;
	row3.push_back(v34);

	matrixV.push_back(row3);

	list<double> row4;
	row4.push_back(v14);
	row4.push_back(v24);
	row4.push_back(v34);
	row4.push_back((invq11 + invq10 + invq01 + invq00)*pdfpdXi4*pdfpdXi4);

	matrixV.push_back(row4);

	//set up the inverse of the matrix - perhaps this can be done better within the calculating inverse method
	row1.clear(); row2.clear(); row3.clear(); row4.clear();

	row1.push_back(1); row1.push_back(0); row1.push_back(0); row1.push_back(0);
	row2.push_back(0); row2.push_back(1); row2.push_back(0); row2.push_back(0);
	row3.push_back(0); row3.push_back(0); row3.push_back(1); row3.push_back(0);
	row4.push_back(0); row4.push_back(0); row4.push_back(0); row4.push_back(1);

	inverse.push_back(row1);
	inverse.push_back(row2);
	inverse.push_back(row3);
	inverse.push_back(row4);

	//calculate inverse of V
	getInverseMatrix(matrixV, inverse);

	//sum all elements of inverse of V to give variance (1^T V^(-1) 1)^(-1)
	unsigned int rowCounter = 0;//need to start counting from 0 for array.

	//loop thro' rows
	for(list< list<double> >::const_iterator r = inverse.begin(); r != inverse.end(); ++r)	
	{

		//loop thro' columns
		for(list<double>::const_iterator c = r->begin(); c != r->end(); ++c)		
		{			
			rowTotals[rowCounter] += *c;
		};

		++rowCounter;
	};

	//not including the 1/total scalar
	totalInverseV = 1.0/(rowTotals[0]+rowTotals[1]+rowTotals[2]+rowTotals[3]);

};

//! Calculates the statistic.
void JointEffects::evaluateStatistic(unsigned int & snp1No, unsigned int & snp2No)
{
	double lambdaA; //cases
	double muA = 0;
	double lambdaN, muN; //controls
	double caseControlStat, diff;
	double caseOnlyStat;
	bool halfAddedCases2ndTime = false;
	bool useAltStatCases = false;
	bool useAltStatCaseCons = false;
	double i22ca, i21ca, i12ca, i11ca, totalInverseVca;
	double rowTotalsca[4] = {0,0,0,0};

	//control variables
	double i22co, i21co, i12co, i11co, totalInverseVco;
	double rowTotalsco[4] = {0,0,0,0};

	//adjust the genotype counts for cases to account for zeroes and if the baseline freq. is too low (< 0.01)
	caseJointGenotypeCounts.setAdjCounts(false);

	calculateis(i22ca, i21ca, i12ca, i11ca, caseJointGenotypeCounts);

	useAltStatCases = (i11ca <= 0.5);

	//calculate variables needed to calculate the case-only stat
	calculateVariables(i22ca, i21ca, i12ca, i11ca, totalInverseVca, rowTotalsca, caseJointGenotypeCounts, useAltStatCases);

	//calculate case only statistic 
	if(useAltStatCases)
	{		
		muA = (rowTotalsca[0]*log((sqrt(i22ca) + 1)*0.5) + rowTotalsca[1]*log((i21ca + 1)*0.5) + rowTotalsca[2]*log((i12ca + 1)*0.5) + rowTotalsca[3]*log(i11ca));

		caseOnlyStat = muA*muA*totalInverseVca;
	}
	else
	{
		//use w's to calculate lambda
		lambdaA = (rowTotalsca[0]*log(i22ca)*0.5 + rowTotalsca[1]*log(i21ca) + rowTotalsca[2]*log(i12ca) + rowTotalsca[3]*log(2*i11ca - 1));

		caseOnlyStat = lambdaA*lambdaA*totalInverseVca;	
	};
	
	//adjust the genotype counts for controls to account for zeroes and if the baseline freq. is too low (< 0.01)
	controlJointGenotypeCounts.setAdjCounts(caseJointGenotypeCounts.halfAdded);

	
	//if 0.5 was added to the control counts then we also need to add 0.5 to the case counts if not already done so
	if(controlJointGenotypeCounts.halfAdded && !caseJointGenotypeCounts.halfAdded)
	{
		halfAddedCases2ndTime = true;
		caseJointGenotypeCounts.setAdjCounts(true);
		calculateis(i22ca, i21ca, i12ca, i11ca, caseJointGenotypeCounts);
	};
	
	//calculate i's for controls
	calculateis(i22co, i21co, i12co, i11co, controlJointGenotypeCounts);

	useAltStatCaseCons = (i11co <= 0.5) || (i11ca <= 0.5);

	//case variables need to be recalculated if counts were adjusted or the alt stat is to be used where it was not previously (or vice versa)
	if(halfAddedCases2ndTime || (useAltStatCaseCons && !useAltStatCases) || (!useAltStatCaseCons && useAltStatCases))
	{
		
		if(!halfAddedCases2ndTime && ((useAltStatCaseCons && !useAltStatCases) || (!useAltStatCaseCons && useAltStatCases)))
		{
			calculateis(i22ca, i21ca, i12ca, i11ca, caseJointGenotypeCounts);
		};

		//reset row totals for recalculation
		rowTotalsca[0] = 0; rowTotalsca[1] = 0; rowTotalsca[2] = 0; rowTotalsca[3] = 0;
		//now recalculate the case variables with the new adjusted counts or for the alt stat for the first time
		calculateVariables(i22ca, i21ca, i12ca, i11ca, totalInverseVca, rowTotalsca, caseJointGenotypeCounts, useAltStatCaseCons);
		recalced++;
	};

	
	//calculate variables for controls, either lambda or mu calcs, (and redo cases if nec.)

	//calculate the variables needed to calculate the case-control stat from the control data
	calculateVariables(i22co, i21co, i12co, i11co, totalInverseVco, rowTotalsco, controlJointGenotypeCounts, useAltStatCaseCons);
	

	if(useAltStatCaseCons)
	{		
		muN = (rowTotalsco[0]*log((sqrt(i22co) + 1)*0.5) + rowTotalsco[1]*log((i21co + 1)*0.5) + rowTotalsco[2]*log((i12co + 1)*0.5) + rowTotalsco[3]*log(i11co));

		if(halfAddedCases2ndTime || !useAltStatCases)
		{
			muA = (rowTotalsca[0]*getLog((sqrt(i22ca) + 1)*0.5) + rowTotalsca[1]*getLog((i21ca + 1)*0.5) + rowTotalsca[2]*getLog((i12ca + 1)*0.5) + rowTotalsca[3]*getLog(i11ca));
		};

		diff = (muA*totalInverseVca - muN*totalInverseVco);
		caseControlStat = diff*diff/(totalInverseVca + totalInverseVco);
	}
	else
	{
		//use w's to calculate lambda
		lambdaN = (rowTotalsco[0]*log(i22co)*0.5 + rowTotalsco[1]*log(i21co) + rowTotalsco[2]*log(i12co) + rowTotalsco[3]*log(2*i11co - 1));

		if(halfAddedCases2ndTime || useAltStatCases)
		{
			lambdaA = (rowTotalsca[0]*log(i22ca)*0.5 + rowTotalsca[1]*log(i21ca) + rowTotalsca[2]*log(i12ca) + rowTotalsca[3]*log(2*i11ca - 1));
		};

		diff = (lambdaA*totalInverseVca - lambdaN*totalInverseVco);
		caseControlStat = diff*diff/(totalInverseVca + totalInverseVco);
		
	};

	if(thresholdCC == 0 || thresholdCO == 0 || caseControlStat >= thresholdCC || caseOnlyStat >= thresholdCO)
	{
               
		totalResults++;
		if(totalResults > maxNoResults && maxNoResults != 0)
		{
			outErr("Total number of results exceed maximum allowed("); outErr(maxNoResults); outErr(")!\n");
			exit(0);
		};
 
		//output alternative variables if used, alt variance already set
		if(useAltStatCases) lambdaA = muA;
		if(useAltStatCaseCons) lambdaN = muN;

		resultsFile << snp1No << " " << descSNPs->getSNPName(snp1No, 1) << " " <<  snp2No << " " << descSNPs->getSNPName(snp2No, 2) << " "
			<< lambdaA << " " << sqrt(totalInverseVca) << " " << lambdaN << " " << sqrt(totalInverseVco) << " " 
			<< caseControlStat << " " << getPvalueChiSq1DF(caseControlStat) << " "
			<< caseOnlyStat << " " << getPvalueChiSq1DF(caseOnlyStat) << "\n";			
	};

	return;
};

//! Inverts a matrix. the inverse should be set to the identity when given to this function
void getInverseMatrix(list< list<double> > & matrix, list< list<double> > & inverse)
{

	//loop thro' rows of matrix, m, and the inverse, i
	list< list<double> >::iterator mrow = matrix.begin();
	list< list<double> >::iterator irow = inverse.begin();
	list< list<double> >::iterator mrow2;
	list< list<double> >::iterator irow2;
	list<double>::iterator mcol;
	list<double>::iterator icol;
	list<double>::iterator mcol2;
	list<double>::iterator icol2;
	double factor;
	unsigned int rowNo = 1;
	unsigned int colNo = 1;

	for( ; mrow != matrix.end(); ++mrow, ++irow)
	{
		//set column to the first column in the row
		mcol = mrow->begin();
		icol = irow->begin();
		colNo = 1;

		//advance the column until the the row no. is equal to the column no.
		while(colNo != rowNo)
		{
			mcol++;			
			++colNo;
		};


		//divide the row in (m and i) by the value in the matrix, m, at (rowNo, colNo)
		factor = 1.0/(*mcol); //divide all elements instead?
		*mcol = 1;		
		mcol++;		

		//scale the remaining elements in the row - if there are any
		while(mcol != mrow->end())
		{
			*mcol *= factor;			
			mcol++;			
		};

		//scale all of the elements in the inverse for this row
		while(icol != irow->end())
		{			
			*icol *= factor;			
			icol++;
		};

		//subtract the row in question away from the remaining rows scaled  s.t. column = mrow will be zero below this row in matrix m
		mrow2 = mrow;
		irow2 = irow;
		mrow2++;
		irow2++;
		//loop thro' remaining rows
		while(mrow2 != matrix.end())
		{
			//set column iterators to begining of the rows
			mcol2 = mrow2->begin();
			icol2 = irow2->begin();
			mcol = mrow->begin();
			icol = irow->begin();

			//advance column of matrix, m, to the same as the main row being considered, with value rowNo
			colNo = 1;
			while(colNo != rowNo)
			{
				mcol++;
				mcol2++;				
				++colNo;
			};

			factor = *mcol2; //factor to multiple row, rowNo by to take away from subseq. rows
			*mcol2 -= (*mcol)*factor;//0;
			mcol++;
			mcol2++;

			//subtract scaled row for the rest of the matrix, m
			while(mcol2 != mrow2->end())
			{
				*mcol2 -= (*mcol)*factor;				
				mcol++;
				mcol2++;				
			};

			//now perform the same row operation on the inverse matrix, i
			while(icol2 != irow2->end())
			{
				*icol2 -= (*icol)*factor;
				icol++;
				icol2++;				
			};

			mrow2++;
			irow2++;
		};//end of performing row operations to set column (=rowNo) to zeroes below row = rowNo

		++rowNo;
	};//end of performing row operations to set lower left of matrix, m, to zeroes

	//Now reduce the upper right of matrix, m, to zero
	list< list<double> >::reverse_iterator mrowre = matrix.rbegin();
	list< list<double> >::reverse_iterator irowre = inverse.rbegin();
	list< list<double> >::reverse_iterator mrowre2 = matrix.rbegin();
	list< list<double> >::reverse_iterator irowre2 = inverse.rbegin();
	list<double>::reverse_iterator mcolre2;
	list<double>::reverse_iterator mcolre;

	rowNo = matrix.size();

	for( ; mrowre != matrix.rend(); ++mrowre, ++irowre)
	{

		mrowre2 = mrowre;
		irowre2 = irowre;
		mrowre2++;
		irowre2++;

		//loop tho' the remaining rows backwards - if there are any
		while(mrowre2 != matrix.rend())
		{			
			//set column iterators to begining of the rows
			mcolre2 = mrowre2->rbegin();
			icol2 = irowre2->begin();
			mcolre = mrowre->rbegin();
			icol = irowre->begin();

			//advance column of matrix, m, to the same as the main row being considered, with value rowNo
			colNo = mrowre2->size();//size will be 4
			while(colNo != rowNo)
			{
				mcolre++;
				mcolre2++;				
				--colNo;
			};

			factor = *mcolre2; //factor to multiple row, rowNo by to take away from subseq. rows
			*mcolre2 -= (*mcolre)*factor;//0;
			mcolre++;
			mcolre2++;

			//subtract scaled row from the rest of the matrix, m
			while(mcolre2 != mrowre2->rend())//could stop at when col < row
			{
				*mcolre2 -= (*mcolre)*factor;				
				mcolre++;
				mcolre2++;				
			};

			//now perform the same row operation on the inverse matrix, i
			while(icol2 != irowre2->end())
			{
				*icol2 -= (*icol)*factor;
				icol++;
				icol2++;				
			};			

			mrowre2++;
			irowre2++;
		};

		--rowNo;
	};

};

double pvalDiff(double & pval, const double & x)
{
	return fabs(pval - 1.0 +  erf(sqrt(x)*0.7071067811865475244008443621048490392));
};

// Calculate the Chi sq value (with 1 df) corresponding to a p-value using a Golden section search.
double calculateChiSqFromPvalue(double & pval)
{
	if(pval > 1) pval = 1;
	if(pval < 0) pval = 0;
	if(pval == 0) return 100000000;
	if(pval == 1) return 0;
	double a = 0;
	double c = 71;
	double b = (a + c)/2;
	double chisqError = 0.000001;
	unsigned int maxSteps = 200;
	double golden = 0.381966012;
	double x, olda, oldb, oldc;
	unsigned int steps = 0;
	double fb = pvalDiff(pval, b);
	double fx;

	do{
		olda = a; oldb = b; oldc = c;
		
		//choose next trial point, x
		if((c-b) > (b-a))
		{
			x = b + golden*(c-b);

			fx = pvalDiff(pval, x);
			if(fx > fb)
			{
				c = x;
			}
			else
			{
				a = b; b = x;
			};
		}
		else
		{
			x = b - golden*(b-a);

			fx = pvalDiff(pval, x);
			if(fx > fb)
			{
				a = x;
			}
			else
			{
				c = b; b = x; 
			};
		};

		fb = pvalDiff(pval, b);

		++steps;
		
	}while(c - a > chisqError && steps <= maxSteps);

	return (c+a)/2;
};
