/************************************************************************
 * CASSI, version 1.01
 * Copyright 2012,
 * Richard Howey
 * Institute of Genetic Medicine, Newcastle University
 *
 * richard.howey@ncl.ac.uk
 * http://www.staff.ncl.ac.uk/richard.howey/
 *
 * This file is part of CASSI, the SNP interaction analysis program.
 *
 * CASSI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * CASSI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CASSI.  If not, see <http://www.gnu.org/licenses/>.
 ************************************************************************/


/*! \file Statistics.h
    \brief This file contains the models used for logistic regression.
    
*/

#ifndef __MODEL
#define __MODEL


#include <map>

using namespace std;

#include "Data.h"

// General Class for Statistics for interaction between two SNPs.
class Statistic
{
protected:

	SNPWindow * window1; //these SNP objects belong to the correspoding SNP window
	SNPWindow * window2;
	DescriptionOfSNPs * descSNPs;
	JointGenotypeCounts caseJointGenotypeCounts;
	JointGenotypeCounts controlJointGenotypeCounts;
	list<bool> caseControls; //list of bool values to whether a subject is a case or not
	unsigned int totalNoSubjects;

	double thresholdCC;
	double thresholdCO;
	unsigned int maxNoResults;
	unsigned int totalResults;
	ofstream resultsFile;
	static const double oneOverSqRoot2;

public:

	Statistic(string & filename, string & filename2, string & resultsFilename, double & thcc, double & thco, unsigned int & mx, DescriptionOfSNPs * desc) : thresholdCC(thcc), thresholdCO(thco), maxNoResults(mx), totalResults(0), descSNPs(desc)		
	{	
		//open results file
		resultsFile.open(resultsFilename.c_str());
		//setup which subject are cases and which are controls
		setupCaseControls(filename);
		checkCaseControls(filename2, filename);
		totalNoSubjects = caseControls.size();			
	};
	
	virtual ~Statistic()
	{
		resultsFile.close();
	};

	void setSNPWindows(SNPWindow * w1, SNPWindow * w2) {window1 = w1; window2 = w2;};
	void updateJointGenotypeCounts();
	void setupCaseControls(string & filename);
	void checkCaseControls(string & filename2, string & filename);
	unsigned int getTotalNoSubjects() const {return totalNoSubjects;};
	virtual void evaluateStatistic(unsigned int & snp1No, unsigned int & snp2No) const {};
	double getPvalueChiSq1DF(double & chisq) const;
	void outputTotalResults() {out("Number of results found: "); out(totalResults); out("\n\n");};
};

void getInverseMatrix(list< list<double> > & matrix, list< list<double> > & inverse);

//! Calculate the Chi sq value (with 1 df) corresponding to a p-value.
double calculateChiSqFromPvalue(double & pval);

//! Joint Effects statistic by Ueki and Cordell
class JointEffects : public Statistic
{
private:

	map<double, double> logCache;
	unsigned int recalced;

public:

	JointEffects(string & filename, string & filename2, string & resultsFilename, double & thcc, double & thco, unsigned int & mx, DescriptionOfSNPs * desc) : Statistic(filename, filename2, resultsFilename, thcc, thco, mx, desc)
	{		
		recalced = 0;
		resultsFile << "SNP1 SNP1_Name SNP2 SNP2_Name "
			        << "Case_logOR Case_SE Con_logOR Con_logSE "
					<< "Case-Con_Chisq Case-Con_Pval Case-Only_Chisq Case-Only_Pval\n";
	};
	
	~JointEffects() {};

	void evaluateStatistic(unsigned int & snp1No, unsigned int & snp2No);
	void calculateis(double & i22, double & i21, double & i12, double & i11, JointGenotypeCounts & jc);
	void calculateVariables(double & i22, double & i21, double & i12, double & i11, double & totalInverseV, double rowTotals[], JointGenotypeCounts & jc, const bool & useAltStat);
	double getOddRatioRelRisk(double & num1, double & num2, double & denom1, double & denom2);
	double getLog(double v);
	void outputCacheSize() {out("Log Cache size: "); out(logCache.size()); out("\n");};
	void outputRecalc() {out("Case variables recalced: "); out(recalced); out("\n");};
};


#endif
