/************************************************************************
 * CASSI, version 1.01
 * Copyright 2012,
 * Richard Howey
 * Institute of Genetic Medicine, Newcastle University
 *
 * richard.howey@ncl.ac.uk
 * http://www.staff.ncl.ac.uk/richard.howey/
 *
 * This file is part of CASSI, the SNP interaction analysis program.
 *
 * CASSI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * CASSI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CASSI.  If not, see <http://www.gnu.org/licenses/>.
 ************************************************************************/


/*! \file main.cpp
    \brief This file reads in the initial input files and options.
    
    This file also outputs usage instructions and program details.
*/

#include <iostream>
#include <ostream>
#include <time.h>
#include <string>

using namespace std; // initiates the "std" or "standard" namespace
 
#include "main.h"
#include "Analysis.h"
#include "Statistics.h"

bool outputToScreen = true; 

ofstream logFile;

//! Output program title to screen
void header()
{
	out("\nCASSI: SNP interaction analysis software, v1.01\n");
	out("----------------------------------------------\n");
	out("Copyright 2012 Richard Howey, GNU General Public License, v3\n");
	out("Institute of Genetic Medicine, Newcastle University\n\n");
};

//! Output program usage to screen
void usage()
{
		header();
	 	
		out("Usage:\n\t ./cassi [options] file.bed\n");
		out(" or ./cassi parameterfile.pf [file.bed]\n\n");

		out("Options:\n");
		out("  -snp1 a1 a2   -- First SNP window, a1 = Start SNP no., a2 = End SNP no.\n");
		out("  -snp2 b1 b2   -- Second SNP window, b1 = Start SNP no., b2 = End SNP no.\n");
		out("  -i file.bed   -- Input file\n");
		out("  -i2 file2.bed -- Second (optional) input file for second SNP window\n");
		out("  -o file.out   -- Output file\n");
		out("  -log file.log -- Output log file\n");	
		out("  -max m        -- Maximum no. of results\n");
		//out("  -pf options   -- Options parameter filename\n");	
		out("  -so           -- suppress output to screen\n\n");

		out("Joint Effects Test Statistic Options:\n");
		out("  -thcc t       -- P-value threshold for case/control test (set to 0 for no output)\n");
		out("  -thco t       -- P-value threshold for case only test (set to 0 for no output)\n");
		out("  -th t         -- P-value threshold for either test (set to 0 for no output)\n\n");

		out("Default Options:\n");
		out("  -snp1 a1 a2   -- All SNPs\n");
		out("  -snp2 b1 b2   -- All SNPs\n");
		out("  -o filename   -- cassi.out\n");
		out("  -log cassi.log\n");	
		out("  -th 0.0001\n");
		out("  -max 1000000  -- (10^6)\n\n");	
		
};

//! Get an option value either from the parameter file or the command line
void getOptionValue(unsigned int & anUnInt, bool & useParaFile, int & argcount, int & argc, char * argv[], ifstream & readParaFile)
{	
	if(useParaFile)
	{
		if(readParaFile.eof()) return;
		readParaFile >> anUnInt;		
	}
	else
	{
		argcount++; if(argcount >= argc) return;				
		anUnInt = atoi(argv[argcount]);		
	};
};

//! Get an option value either from the parameter file or the command line
void getOptionValue(double & aDouble, bool & useParaFile, int & argcount, int & argc, char * argv[], ifstream & readParaFile)
{
	if(useParaFile)
	{
		if(readParaFile.eof()) return;		
		readParaFile >> aDouble;		
	}
	else
	{
		argcount++; if(argcount >= argc) return;			
		aDouble = atof(argv[argcount]);		
	};
};

//! Get an option value either from the parameter file or the command line
void getOptionValue(string & aString, bool & useParaFile, int & argcount, int & argc, char * argv[], ifstream & readParaFile)
{
	if(useParaFile)
	{
		if(readParaFile.eof()) return;		
		readParaFile >> aString;
	}
	else
	{
		argcount++; if(argcount >= argc) return;		
		aString = argv[argcount];
	};
};

//! The start of the program
int main(int argc, char * argv[])
{
	time_t start,end;
	double dif;
	time(&start);

	int argcount = 1;
	string option = "";
	string filename = "";
	string filename2 = "";
	string paraFilename = "";
	string outputFileName = "cassi.out";
	string logFilename = "";

	unsigned int snp1StartSNP = 1;
	unsigned int snp1EndSNP = 0;
	unsigned int snp2StartSNP = 1;
	unsigned int snp2EndSNP = 0;
	outputToScreen = true;	
	double thresholdCC = 1e-4;
	double thresholdCO = 1e-4;
	double threshold = -1;
	unsigned int maxNoResults = 1000000;
	bool useParaFile = false;
	ifstream readParaFile;
	
	if(argcount < argc) option = argv[argcount];

	//deal with parameter file
	if(option == "-pf" || (option.length() >= 3 && option.substr(option.length()-3) == ".pf"))
	{
		if(option == "-pf")
		{
			argcount++; 
			if(argcount < argc) paraFilename = argv[argcount];
		}
		else paraFilename = option;

		//open parameter file		
		readParaFile.open(paraFilename.c_str());
		if(!readParaFile.is_open())
		{
			header();
			outErr("Cannot read parameter file: "); outErr(paraFilename); outErr("!\n");
			exit(0);
		};

		argcount++; 
		useParaFile = true;
	};

	
	//set given options
	while((!useParaFile && argcount < argc && argv[argcount][0] == '-') || (useParaFile && !readParaFile.eof()))
	{

		if(useParaFile)
		{
			//find the start of the next command
			do{
				readParaFile >> option;
				if(option.length() >= 2 && option.substr(0,1) == "-") break;
			}while(!readParaFile.eof());
		}
		else
		{
			option = argv[argcount];
		};

		if(useParaFile && readParaFile.eof()) break;

		if(option ==  "-snp1")
		{	
			getOptionValue(snp1StartSNP, useParaFile, argcount, argc, argv, readParaFile);
			getOptionValue(snp1EndSNP, useParaFile, argcount, argc, argv, readParaFile);						
		}
		else if(option ==  "-snp2")
		{	
			getOptionValue(snp2StartSNP, useParaFile, argcount, argc, argv, readParaFile);
			getOptionValue(snp2EndSNP, useParaFile, argcount, argc, argv, readParaFile);			
		}
		else if(option ==  "-i") getOptionValue(filename, useParaFile, argcount, argc, argv, readParaFile);					
		else if(option ==  "-i2") getOptionValue(filename2, useParaFile, argcount, argc, argv, readParaFile);		
		else if(option ==  "-o") getOptionValue(outputFileName, useParaFile, argcount, argc, argv, readParaFile);					
		else if(option ==  "-log") getOptionValue(logFilename, useParaFile, argcount, argc, argv, readParaFile);
		else if(option ==  "-thcc") getOptionValue(thresholdCC, useParaFile, argcount, argc, argv, readParaFile);	
		else if(option ==  "-thco") getOptionValue(thresholdCO, useParaFile, argcount, argc, argv, readParaFile);	
		else if(option ==  "-th") getOptionValue(threshold, useParaFile, argcount, argc, argv, readParaFile);						
		else if(option ==  "-max") getOptionValue(maxNoResults, useParaFile, argcount, argc, argv, readParaFile);						
		else if(option == "-so") outputToScreen = false;
		else
		{
			if(logFilename != "") logFile.open(logFilename.c_str());
			else logFile.open("cassi.log");

			header();
    		if(useParaFile) {outErr("Unrecognised option: "); outErr(option); outErr("\n\n");}
			else {outErr("Unrecognised command line switch: "); outErr(option); outErr("\n\n");};			
    		exit(0);
		};

		if(!useParaFile) argcount++;
	};

	if(argcount < argc) filename = argv[argcount];	
	
	if(logFilename != "") logFile.open(logFilename.c_str());
	else logFile.open("cassi.log");

	//set both thresholds
	if(threshold != -1)
	{
		thresholdCC = threshold;
		thresholdCO = threshold;
	};

	if(argc==1)
	{
		usage();
		exit(0);
	}
	else if(filename == "")
	{
		outErr("\nInput file not set!\n\n");
		usage();
		exit(0);
	}
	else if(filename.length() >= 4 && filename.substr(filename.length()-4) != ".bed" ||
		 filename2.length() >= 4 && filename2.substr(filename2.length()-4) != ".bed")
	{
		header();
		outErr("A binary pedigree file (.bed) is required for CASSI!\n\n");
		outErr("Try using PLINK with\n");
		outErr("plink --noweb --file mydata --make-bed\n\n");
		exit(0);
	};

	//output options to screen
	header();
	out("Parameters:\n");
	out("Input file: "); out(filename); out("\n");
	if(filename2 != "") {out("Second input file: "); out(filename2); out("\n");};
	out("Output file: "); out(outputFileName); out("\n");
		
	out("Start SNP of first SNP window: "); out(snp1StartSNP); out("\n");
	if(snp1EndSNP != 0) {out("End SNP of first SNP window: "); out(snp1EndSNP); out("\n");}
	else out("End SNP of first window is the last SNP\n");

	out("Start SNP of second SNP window: "); out(snp2StartSNP); out("\n");
	if(snp2EndSNP != 0) {out("End SNP of second SNP window: "); out(snp2EndSNP); out("\n");}
	else out("End SNP of second window is the last SNP\n");

	if(maxNoResults == 0) out("Maximum no. of results: no maximum\n");
	else {out("Maximum no. of results: "); out(maxNoResults); out("\n\n");};

	out("Test Statistic: Joint Effects\n");
	out("P-value threshold for case/control results: "); out(thresholdCC); out("\n");
	out("P-value threshold for case only results: "); out(thresholdCO); out("\n");
	
	out("\n");
	
	//create analysis option and run analysis
	Analysis anAnalysis(filename, filename2, outputFileName, snp1StartSNP, snp1EndSNP, snp2StartSNP, snp2EndSNP, thresholdCC, thresholdCO, maxNoResults);

	anAnalysis.runAnalysis();
	logFile.close();

	time(&end);
	dif = difftime(end, start);
	out("Run time: "); out(getTime(dif)); out("\n\n");
};

