/************************************************************************
 * PseudoCons, version 1.0
 * Copyright 2013,
 * Richard Howey
 * Institute of Genetic Medicine, Newcastle University
 *
 * richard.howey@ncl.ac.uk
 * http://www.staff.ncl.ac.uk/richard.howey/
 *
 * This file is part of PseudoCons.
 *
 * PseudoCons is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PseudoCons is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PseudoCons.  If not, see <http://www.gnu.org/licenses/>.
 ************************************************************************/


/*! \file Data.cpp
    \brief This file contains functions for processing basic genotype infomation.
       
*/

#include "Data.h"
#include "main.h"
#include "Analysis.h"

#include <string>
#include <map>
#include <set>
#include <math.h>
#include <iostream>
#include <fstream>
#include <ostream>
#include <sstream>
#include <cstdlib>

using namespace std; // initiates the "std" or "standard" namespace


//! Get the ID of a subject, family ID + subject ID maps to an ordinal number over all subjects. 
unsigned int MapIds::getId(const string & name)
{
  map<string, unsigned int>::const_iterator i = idMap.find(name);
  if(i != idMap.end()) return i->second;

  //if id does not exist create an id
  unsigned int id = (unsigned)idMap.size() + 1;
  idMap[name] = id;
  
  return id;
};

//! Get the name of a subject, given the id.
string MapIds::getName(const unsigned int & id)
{
	//loop thro' names to find the id, this is not too quick but is only used for reporting errors/warnings
	for(map<string, unsigned int>::const_iterator i = idMap.begin(); i != idMap.end(); ++i)
	{
		if(id == i->second) return i->first;
	};

	return "ID not found";
};

//! Adds a subject to the pedigree.
void AllPedigreeData::addSubjectToPedigree(unsigned int & subjectId, Subject * subject, unsigned int & pedigreeId, string & pedIdName)
{
	map<unsigned int, Pedigree *>::iterator p = pedigrees.find(pedigreeId);

	//if pedigree exists add the subject to it otherwise add subject to a new pedigree
	if(p != pedigrees.end())
	{
		p->second->addSubject(subjectId, subject);
	}
	else
	{
		Pedigree * pedigree = new Pedigree(pedigreeId, pedIdName);
		pedigree->addSubject(subjectId, subject);
		pedigrees[pedigreeId] = pedigree;
	};

};

//! Outputs basic stats of the pedigree to screen.
void AllPedigreeData::outputSummary()
{
	double noOfPedigrees = (double)pedigrees.size();
	double meanPedigreeSize = 0;
	double standardDeviationPedigreeSize = 0;

	for(map<unsigned int, Pedigree *>::const_iterator ped = pedigrees.begin(); ped != pedigrees.end(); ++ped)
	{
		meanPedigreeSize += ped->second->getNumberOfSubjects();
	};
	meanPedigreeSize /= noOfPedigrees;

	for(map<unsigned int, Pedigree *>::const_iterator ped = pedigrees.begin(); ped != pedigrees.end(); ++ped)
	{
		standardDeviationPedigreeSize += (ped->second->getNumberOfSubjects()- meanPedigreeSize)*(ped->second->getNumberOfSubjects()- meanPedigreeSize);
	};
	standardDeviationPedigreeSize /= (noOfPedigrees - 1);
	standardDeviationPedigreeSize = sqrt(standardDeviationPedigreeSize);

	out("Number of pedigrees: "); out(noOfPedigrees); out("\n");
	out("Mean pedigree size: "); out(meanPedigreeSize); out("\n");
	out("Standard deviation of pedigree size: "); out(standardDeviationPedigreeSize); out("\n\n");
};

//! Unmarks subjects in pedigree, used when allowing extras trios.
void AllPedigreeData::restoreAllSubjectsToAllPedigrees()
{
	for(map<unsigned int, Pedigree *>::iterator p = pedigrees.begin(); p != pedigrees.end(); ++p)
	{
		p->second->restoreAllSubjectsToPedigree();
	};

};

//! checks if a subject exists and if it does whether it is affected or not
void AllPedigreeData::subjectExistsAndAffected(const unsigned int & pedId, const unsigned int & subjectId, MapIds & subjectIds) 
{
	bool exists = false;
	bool affected = false;
	bool fatherExists = false;
	bool motherExists = false;

	Subject * subject;
	
	for(map<unsigned int, Pedigree *>::iterator p = pedigrees.begin(); p != pedigrees.end(); ++p)
	{
		subject = p->second->getSubject(subjectId, exists);
		if(exists)
		{
			affected = subject->getAffected();
			fatherExists = subject->getFatherId() != 0;
			motherExists = subject->getMotherId() != 0;
			break;
		};
	};

	if(!exists || !affected || !fatherExists || !motherExists)
	{
		string subjectName = subjectIds.getName(subjectId);
		
		if(!exists) {outErr("Warning: proband subject "); outErr(subjectName); outErr(" does not exist!\n");}
		else if(!affected) {outErr("Warning: proband subject "); outErr(subjectName); outErr(" is not affected!\n");}
		else if(!motherExists) {outErr("Warning: proband subject "); outErr(subjectName); outErr(" has no mother!\n");}
		else if(!fatherExists) {outErr("Warning: proband subject "); outErr(subjectName); outErr(" has no father!\n");};
	};
	
};

//! Adds trio to list and updates pseudocontrol basic info.
void AllTrioData::addTrio(Trio * aTrio)
{
	trios.push_back(aTrio);	
};	

//! Creates pseudocontrols for each trio.
void AllTrioData::createPseudoControls(MapIds & subjectIds, ofstream & psFamFile, bool & interaction)
{
	for(list<Trio *>::iterator t = trios.begin(); t != trios.end(); ++t)
	{
		(*t)->createPseudoControls(noPseudoCons, subjectIds, psFamFile, interaction);
	};
};

//! Updates SNP info for each pseudocontrol.
void AllTrioData::updatePsuedoControlSNPData(ofstream & psBedFile, unsigned int & psBitCount, int & aBit)
{
	for(list<Trio *>::iterator t = trios.begin(); t != trios.end(); ++t)
	{
		(*t)->updatePsuedoControlSNPData(noPseudoCons, psBedFile, psBitCount, aBit);
	};

};

//! Write last byte and resets the Bit and bit counter
void writeLastByteBeforeNextSNP(ofstream & psBedFile, unsigned int & psBitCount, int & aBit)
{
	if(psBitCount == 0) return; //last byte may already be written if the number of pseudocontrols is a multiple of 4

	//write to file
	char buffer[1];
	buffer[0] = aBit;
	psBedFile.write(buffer, 1);

	psBitCount = 0;
	aBit = 0;
};

//! Updates joint SNP info for each pseudocontrol.
void AllTrioData::updatePsuedoControlJointSNPData(ofstream & psBedFile, unsigned int & psBitCount, int & aBit)
{

	for(list<Trio *>::iterator t = trios.begin(); t != trios.end(); ++t)
	{
		(*t)->updatePsuedoControlJointSNPData(noPseudoCons, psBedFile, psBitCount, aBit);
	};

	//start new byte as starting a new SNP
	writeLastByteBeforeNextSNP(psBedFile, psBitCount, aBit);

	for(list<Trio *>::iterator f = trios.begin(); f != trios.end(); ++f)
	{
		(*f)->writePseudoControlSecondSNPData(psBedFile, psBitCount, aBit);
	};
};

//! Outputs info about pseudocontrols to screen and log file.
void AllTrioData::outputPseudoControlInfo()
{
	list<Subject * > pseudoControls;
	unsigned int maleCountPseudo = 0;
	unsigned int femaleCountPseudo = 0;
	unsigned int unknownCountPseudo = 0;
	unsigned int maleCountCase = 0;
	unsigned int femaleCountCase = 0;
	unsigned int unknownCountCase = 0;
	unsigned int sex;

	for(list<Trio *>::iterator t = trios.begin(); t != trios.end(); ++t)
	{
		sex = (*t)->getChild()->getSexId();
		if(sex == 1) maleCountCase++;
		else if(sex == 2) femaleCountCase++;
		else unknownCountCase++;

		pseudoControls = (*t)->getPseudoControls();
		for(list<Subject * >::iterator p = pseudoControls.begin(); p != pseudoControls.end(); ++p)
		{
			sex = (*p)->getSexId();
			if(sex == 1) maleCountPseudo++;
			else if(sex == 2) femaleCountPseudo++;
			else unknownCountPseudo++;
		};
	};

	unsigned int totalNoOfCases = maleCountCase+femaleCountCase+unknownCountCase;

	double femalePercent, malePercent, unknownSexPercent;
	malePercent = (((double)(maleCountCase))/((double)(totalNoOfCases)))*100;
	femalePercent = (((double)(femaleCountCase))/((double)(totalNoOfCases)))*100;
	unknownSexPercent = (((double)(unknownCountCase))/((double)(totalNoOfCases)))*100;
	
	out("Number of cases: "); out(totalNoOfCases);
	out("\n          Males: "); out(maleCountCase); out(" ("); out(malePercent); out("%)");
	out("\n          Females: "); out(femaleCountCase); out(" ("); out(femalePercent); out("%)");
	out("\n          Unknown sex: "); out(unknownCountCase); out(" ("); out(unknownSexPercent); out("%)\n\n");
	
	unsigned int totalNoOfPsCons = maleCountPseudo+femaleCountPseudo+unknownCountPseudo;

	malePercent = (((double)(maleCountPseudo))/((double)(totalNoOfPsCons)))*100;
	femalePercent = (((double)(femaleCountPseudo))/((double)(totalNoOfPsCons)))*100;
	unknownSexPercent = (((double)(unknownCountPseudo))/((double)(totalNoOfPsCons)))*100;
	
	out("Number of pseudocontrols: "); out(totalNoOfPsCons);
	out("\n          Males: "); out(maleCountPseudo); out(" ("); out(malePercent); out("%)");
	out("\n          Females: "); out(femaleCountPseudo); out(" ("); out(femalePercent); out("%)");
	out("\n          Unknown sex: "); out(unknownCountPseudo); out(" ("); out(unknownSexPercent); out("%)\n\n");
	
};

//! Outputs details of a subject to screen.
void Subject::outputDetails()
{
	out("Subject id: "); out(id); out("\n");
	out("Father id: "); out(fatherId); out("\n");
	out("Mother id: "); out(motherId); out("\n");
	out("Sex id: "); out(sex); out("\n");
	out("Affected id: "); out(affected); out("\n\n");
};

//! Returns a subject in a pedigree
Subject * Pedigree::getSubject(const unsigned int & id) const
{
	map<unsigned int, Subject *>::const_iterator s = subjects.find(id);
	if(s != subjects.end()) return s->second;

	outErr("Error: subject not found in pedigree!\n");
	exit(0);	       
};

//! Returns a subject in a pedigree and returns if it exists. 
Subject * Pedigree::getSubject(const unsigned int & id, bool & exists) const
{
	map<unsigned int, Subject *>::const_iterator s = subjects.find(id);
	if(s != subjects.end())
	{
		exists = true;
		return s->second;
	};

	exists = false;
	return 0;       
};

//! Checks if a subject is a child of a case parent trio in a pedigree.
Trio * Pedigree::findCaseParentTrio(bool & found, Subject * subject)
{
	bool fatherExists = false;
	bool motherExists = false;
	
	//check if subject is affected, has genotype data and has not been removed if allowing extra trios 
	if(subject->getAffected() /*&& subject->genotypeDataPresent(snpId)*/ && subject->isNotRemoved())
	{
		//check if parents exist and have genotype data
		if(subject->getFatherId() != 0 && subject->getMotherId() != 0)
		{
			Subject * father = getSubject(subject->getFatherId(), fatherExists);
			Subject * mother = getSubject(subject->getMotherId(), motherExists);
			if(fatherExists && motherExists /*&& father->isNotRemoved() && mother->isNotRemoved()*/)
			{
				//if(father->genotypeDataPresent(snpId) && mother->genotypeDataPresent(snpId))
				{ 
					Trio * aTrio = new Trio(pedIdName, father, mother, subject);					
					found = true;
					return aTrio;
				};
			};
		};
	};

	return 0;
};

//! Loops thro' subjects in pedigree to find a case parent trio.
Trio * Pedigree::findCaseParentTrio(bool & found)
{
	found = false;
	Trio * aTrio;	

	//loop through subjects looking for a case which has parents
	for(map<unsigned int, Subject *>::const_iterator s = subjects.begin(); (s != subjects.end() && !found); ++s)
	{
		 aTrio = findCaseParentTrio(found, s->second);	
	};

	return aTrio;
};

//! Create the pseudocontrols and add basic info but no genotype data yet.
void Trio::createPseudoControls(unsigned int & noPseudoCons, MapIds & subjectIds, ofstream & psFamFile, bool & interaction)
{
	Subject * aPseudoControl;
	string pseudoConIDName;
	unsigned int pseudoConID;
	unsigned int patID = 0, matID = 0;
	unsigned int sexID = child->getSexId();
	string childIDName = child->getIDName();
	bool affected = false; //not affected for control
	
	//output case to fam file
	psFamFile << pedIdName << " " << childIDName << " " << patID << " " << matID << " " << sexID << " 2\n";

	for(unsigned int psNo = 1; psNo <= noPseudoCons; ++psNo)
	{
		pseudoConIDName = childIDName + "-pseudo-" + toString(psNo);
		pseudoConID = subjectIds.getId(pseudoConIDName);		

		if(interaction) aPseudoControl = new SubjectTwoSnps(pseudoConID, patID, matID, sexID, affected, pseudoConIDName);
		else aPseudoControl = new SubjectOneSnp(pseudoConID, patID, matID, sexID, affected, pseudoConIDName);

		pseudoControls.push_back(aPseudoControl);

		//output to fam file
		psFamFile << pedIdName << " " << pseudoConIDName << " " << patID << " " << matID << " " << sexID << " 1\n";
	};
	
};

void writeBedGenotype(ofstream & psBedFile, unsigned int & psBitCount, int & aBit, const bool & allele1, const bool & allele2)
{
	
//	int addOne = '\1';
//	for(unsigned int i = 1; i<=7; ++i) addOne = addOne << 1; //shift bits to the left
//cout << addOne <<" addOne\n";

	aBit = aBit >> 1; //shift bits to the right
	if(allele1) aBit = aBit | 128; 
	aBit = aBit >> 1;  //shift bits to the right
	if(allele2) aBit = aBit | 128;
	
	psBitCount += 2;

	//write to file if byte is finished otherwise move bits over to left
	if(psBitCount == 8)
	{
		//write to file
		char buffer[1];
		buffer[0] = aBit;
		psBedFile.write(buffer, 1);

		psBitCount = 0;
		aBit = 0;
	};

};

//! Checks for a Medelian error
bool checkForMendelianError(pair<bool, bool> & fatherGeno, pair<bool, bool> & motherGeno, pair<bool, bool> & childGeno)
{
	//suppose that first child allele is from father and second from mother, then vice versa
	bool isOK = ((childGeno.first == fatherGeno.first || childGeno.first == fatherGeno.second) && (childGeno.second == motherGeno.first || childGeno.second == motherGeno.second))
		|| ((childGeno.second == fatherGeno.first || childGeno.second == fatherGeno.second) && (childGeno.first == motherGeno.first || childGeno.first == motherGeno.second));

	return !isOK;
};

//! Given trio genotypes returns 1 psuedo control based on the alleles that were not transmitted.
pair<bool, bool> getPseudoControlAllele(pair<bool, bool> & fatherGeno, pair<bool, bool> & motherGeno, pair<bool, bool> & childGeno)
{
	if((fatherGeno.first == 1 && fatherGeno.second == 0) || (motherGeno.first == 1 && motherGeno.second == 0)
		|| (childGeno.first == 1 && childGeno.second == 0) || checkForMendelianError(fatherGeno, motherGeno, childGeno)) //if mother or father data is missing then so is the pseudocontrol
	{
			return make_pair(true, false);
	};
	
	bool psAllele1, psAllele2;

	//add parent genotypes to set and then remove the ones that were transmitted
	multiset<bool> alleles;
	multiset<bool>::const_iterator i;
	alleles.insert(fatherGeno.first);
	alleles.insert(fatherGeno.second);
	alleles.insert(motherGeno.first);
	alleles.insert(motherGeno.second);

	i = alleles.find(childGeno.first);
	if(i != alleles.end()) alleles.erase(i); 
	i = alleles.find(childGeno.second);
	if(i != alleles.end()) alleles.erase(i);

	multiset<bool>::const_iterator a = alleles.begin();
	psAllele1 = *a; a++;
	psAllele2 = *a;

	return make_pair(psAllele1, psAllele2);
};

//! Given trio genotypes returns 3 psuedo controls based on the 3 genotypes that were not transmitted.
list<pair<bool, bool> > getPseudoControlGenotype(pair<bool, bool> & fatherGeno, pair<bool, bool> & motherGeno, pair<bool, bool> & childGeno)
{
	list<pair<bool, bool> > ans;
	
	//add all possible genotypes that could be passed on and except the one that was
	if((fatherGeno.first == 1 && fatherGeno.second == 0) || (motherGeno.first == 1 && motherGeno.second == 0)
		|| (childGeno.first == 1 && childGeno.second == 0) || checkForMendelianError(fatherGeno, motherGeno, childGeno)) //if mother or father data is missing then so is the pseudocontrol
	{
		for(unsigned int k = 1; k <= 3; ++k) ans.push_back(make_pair(true, false));
	}
	else
	{
		//add genotypes in order and do not add the child genotype		
		bool skipped = false;
		pair<bool, bool> aPair;

		unsigned int allele2CountChild = childGeno.first + childGeno.second;
		unsigned int allele2Count = fatherGeno.first + motherGeno.first;

		if(allele2Count != allele2CountChild)
		{
				aPair = make_pair(fatherGeno.first, motherGeno.first);
				if(aPair.first && !aPair.second) aPair = make_pair(false, true);
				ans.push_back(aPair);				
		} else skipped = true;

		allele2Count = fatherGeno.first + motherGeno.second;

		if(skipped || (allele2Count != allele2CountChild))
		{
			    aPair = make_pair(fatherGeno.first, motherGeno.second);
				if(aPair.first && !aPair.second) aPair = make_pair(false, true);
				ans.push_back(aPair);						
		} else skipped = true;

		allele2Count = fatherGeno.second + motherGeno.first;

		if(skipped || (allele2Count != allele2CountChild))
		{
				aPair = make_pair(fatherGeno.second, motherGeno.first);
				if(aPair.first && !aPair.second) aPair = make_pair(false, true);
				ans.push_back(aPair);						
		} else skipped = true;

		allele2Count = fatherGeno.second + motherGeno.second;

		if(skipped || (allele2Count != allele2CountChild))
		{
				aPair = make_pair(fatherGeno.second, motherGeno.second);
				if(aPair.first && !aPair.second) aPair = make_pair(false, true);
				ans.push_back(aPair);				
		};

	};

	return ans;
};

//! Adds SNP data to the psuedo controls based on the trio.
void Trio::updatePsuedoControlSNPData(unsigned int & noPseudoCons, ofstream & psBedFile, unsigned int & psBitCount, int & aBit)
{
	pair<bool, bool> fatherGeno = father->getGenotype();
	pair<bool, bool> motherGeno = mother->getGenotype();
	pair<bool, bool> childGeno = child->getGenotype();	
	list<Subject * >::iterator pc = pseudoControls.begin();

	//write case genotypes firstly
	writeBedGenotype(psBedFile, psBitCount, aBit, childGeno.first, childGeno.second);

	if(noPseudoCons == 1)
	{
		
		if(pc == pseudoControls.end())
		{
			outErr("Failed to create pseudocontrol for subjects ");
			outErr(father->getIDName()); outErr(", ");
			outErr(mother->getIDName()); outErr(" and ");
			outErr(child->getIDName()); outErr("!\n");
			exit(0);
		};
		
		pair<bool, bool> pseudoControlGeno = getPseudoControlAllele(fatherGeno, motherGeno, childGeno);

		(*pc)->addGenotype(pseudoControlGeno.first, pseudoControlGeno.second);
		writeBedGenotype(psBedFile, psBitCount, aBit, pseudoControlGeno.first, pseudoControlGeno.second);
	}
	else if(noPseudoCons == 3)
	{		
		
		list<pair<bool, bool> > pseudoControlsGeno = getPseudoControlGenotype(fatherGeno, motherGeno, childGeno);

		list<pair<bool, bool> >::const_iterator g = pseudoControlsGeno.begin();

		for(unsigned int psNo = 1; psNo <=3; psNo++)
		{
			if(pc == pseudoControls.end())
			{
				outErr("Failed to create pseudocontrol for subjects ");
				outErr(father->getIDName()); outErr(", ");
				outErr(mother->getIDName()); outErr(" and ");
				outErr(child->getIDName()); outErr("!\n");
				exit(0);
			};

			(*pc)->addGenotype(g->first, g->second);
			writeBedGenotype(psBedFile, psBitCount, aBit, g->first, g->second);			

			++pc; ++g;
		};
	};

};

//! Adds SNP data to the psuedo controls based on the trio.
void Trio::updatePsuedoControlJointSNPData(unsigned int & noPseudoCons, ofstream & psBedFile, unsigned int & psBitCount, int & aBit)
{
	pair<bool, bool> fatherGeno = father->getGenotype(1);
	pair<bool, bool> motherGeno = mother->getGenotype(1);
	pair<bool, bool> childGeno = child->getGenotype(1);
	pair<bool, bool> fatherGeno2 = father->getGenotype(2);
	pair<bool, bool> motherGeno2 = mother->getGenotype(2);
	pair<bool, bool> childGeno2 = child->getGenotype(2);

	list<Subject * >::iterator pc = pseudoControls.begin();

	//write first case SNP now and write second SNP later after all the data for the first SNP has been written
	writeBedGenotype(psBedFile, psBitCount, aBit, childGeno.first, childGeno.second);	

	//if(noPseudoCons == 1)
	//{		
	//	if(pc == pseudoControls.end())
	//	{
	//		outErr("Failed to create pseudocontrol for subjects ");
	//		outErr(father->getIDName()); outErr(", ");
	//		outErr(mother->getIDName()); outErr(" and ");
	//		outErr(child->getIDName()); outErr("!\n");
	//		exit(0);
	//	};

	//	//genotype data for SNP 1 and SNP 2
	//	pair<bool, bool> pseudoControlGeno1 = getPseudoControlAllele(fatherGeno, motherGeno, childGeno);		
	//	pair<bool, bool> pseudoControlGeno2 = getPseudoControlAllele(fatherGeno2, motherGeno2, childGeno2);

	//	(*pc)->addGenotype(pseudoControlGeno1.first, pseudoControlGeno1.second, 1);
	//	(*pc)->addGenotype(pseudoControlGeno2.first, pseudoControlGeno2.second, 2);

	//	//write first SNP now and write second SNP later after all the data for the first SNP has been written
	//	writeBedGenotype(psBedFile, psBitCount, aBit, pseudoControlGeno1.first, pseudoControlGeno1.second);
	//}
	//else if(noPseudoCons == 3)
	//{		
	//	
	//	list<pair<bool, bool> > pseudoControlsGeno = getPseudoControlGenotype(fatherGeno, motherGeno, childGeno);

	//	list<pair<bool, bool> >::const_iterator g = pseudoControlsGeno.begin();

	//	for(unsigned int psNo = 1; psNo <=3; psNo++)
	//	{
	//		if(pc == pseudoControls.end())
	//		{
	//			outErr("Failed to create pseudocontrol for subjects ");
	//			outErr(father->getIDName()); outErr(", ");
	//			outErr(mother->getIDName()); outErr(" and ");
	//			outErr(child->getIDName()); outErr("!\n");
	//			exit(0);
	//		};

	//		//genotype data for SNP 1 and SNP 2
	//		pair<bool, bool> pseudoControlGeno1 = getPseudoControlAllele(fatherGeno, motherGeno, childGeno);			
	//		pair<bool, bool> pseudoControlGeno2 = getPseudoControlAllele(fatherGeno2, motherGeno2, childGeno2);

	//		(*pc)->addGenotype(pseudoControlGeno1.first, pseudoControlGeno1.second, 1);
	//		(*pc)->addGenotype(pseudoControlGeno2.first, pseudoControlGeno2.second, 2);

	//		//write first SNP now and write second SNP later after all the data for the first SNP has been written
	//		writeBedGenotype(psBedFile, psBitCount, aBit, pseudoControlGeno1.first, pseudoControlGeno1.second);					

	//		++pc; ++g;
	//	};
	//}
	/*else if(noPseudoCons == 15)
	{*/		
		list<unsigned int > genotypes1; //allele 2 counts
		list<unsigned int > genotypes2;

		//if mother or father data is missing then so is the pseudocontrol 
		if((fatherGeno.first == 1 && fatherGeno.second == 0) || (motherGeno.first == 1 && motherGeno.second == 0)
			|| (fatherGeno2.first == 1 && fatherGeno2.second == 0) || (motherGeno2.first == 1 && motherGeno2.second == 0)
			|| (childGeno.first == 1 && childGeno.second == 0) || (childGeno2.first == 1 && childGeno2.second == 0)
			|| checkForMendelianError(fatherGeno, motherGeno, childGeno) || checkForMendelianError(fatherGeno2, motherGeno2, childGeno2)) 
		{			
			for(unsigned int i = 1; i <= 15; ++i)
			{				
				(*pc)->addGenotype(true, false, 1);
				(*pc)->addGenotype(true, false, 2);

				//write first SNP now and write second SNP later after all the data for the first SNP has been written
				writeBedGenotype(psBedFile, psBitCount, aBit, true, false);
				
				++pc;
			};	

			return;
		};
	
		//add genotype data to lists
		genotypes1.push_back(fatherGeno.first + motherGeno.first);		
		genotypes1.push_back(fatherGeno.first + motherGeno.second);			
		genotypes1.push_back(fatherGeno.second + motherGeno.first);			
		genotypes1.push_back(fatherGeno.second + motherGeno.second);
	
		//add genotype data to lists
		genotypes2.push_back(fatherGeno2.first + motherGeno2.first);		
		genotypes2.push_back(fatherGeno2.first + motherGeno2.second);			
		genotypes2.push_back(fatherGeno2.second + motherGeno2.first);			
		genotypes2.push_back(fatherGeno2.second + motherGeno2.second);
		
		unsigned int allele2CountChild1 = childGeno.first + childGeno.second;
		unsigned int allele2CountChild2 = childGeno2.first + childGeno2.second;	

		//create psudeo controls for each pair of SNPs not transmitted
		bool skipped = false;
		unsigned int pseudoCount = 0;
		for(list<unsigned int >::const_iterator g1 = genotypes1.begin(); g1 != genotypes1.end(); ++g1)
		{
			for(list<unsigned int >::const_iterator g2 = genotypes2.begin(); g2 != genotypes2.end(); ++g2)
			{
				if((!skipped && (*g1 != allele2CountChild1) && (*g2 != allele2CountChild2))
					 || pseudoCount == 15) //last clause is in case of a mendelian error
				{
					skipped = true;
				}
				else
				{					
					if(pc == pseudoControls.end())
					{
						outErr("Failed to create pseudocontrol for subjects ");
						outErr(father->getIDName()); outErr(", ");
						outErr(mother->getIDName()); outErr(" and ");
						outErr(child->getIDName()); outErr("!\n");
						exit(0);
					};

					//write first SNP now and write second SNP later after all the data for the first SNP has been written
					if(*g1 == 2) {(*pc)->addGenotype(true, true, 1); writeBedGenotype(psBedFile, psBitCount, aBit, true, true);}
					else if(*g1 == 1) {(*pc)->addGenotype(false, true, 1); writeBedGenotype(psBedFile, psBitCount, aBit, false, true);}
					else {(*pc)->addGenotype(false, false, 1); writeBedGenotype(psBedFile, psBitCount, aBit, false, false);};					

					if(*g2 == 2) (*pc)->addGenotype(true, true, 2);
					else if(*g2 == 1) (*pc)->addGenotype(false, true, 2);
					else (*pc)->addGenotype(false, false, 2);					
										
					++pc;
					pseudoCount++;
				}; 

			};
		};

	//};

};

//!
void Trio::writePseudoControlSecondSNPData(ofstream & psBedFile, unsigned int & psBitCount, int & aBit)
{
	//write second SNP case info first
	pair<bool, bool> childGeno2 = child->getGenotype(2);
	writeBedGenotype(psBedFile, psBitCount, aBit, childGeno2.first, childGeno2.second);

	pair<bool, bool> aGenotype;
	for(list<Subject * >::iterator p = pseudoControls.begin(); p != pseudoControls.end(); ++p)
	{
		aGenotype = (*p)->getGenotype(2);
		writeBedGenotype(psBedFile, psBitCount, aBit, aGenotype.first, aGenotype.second);	
	};
};

//! Unmark all subjects in pedigree for when adding extra trios.
void Pedigree::restoreAllSubjectsToPedigree()
{
	for(map<unsigned int, Subject *>::const_iterator s = subjects.begin(); s != subjects.end(); ++s)
	{
		s->second->restoreToPedigree();
	};
};

