//////////////////////////////////////////////////////////////////////////
//                              hilbert.c                               //
//                                                                      //
// Purpose                                                              //
// =======                                                              //
// This program generates space-filling curves called Hilbert curves.   //                                                                      //
// Author            Creation Date                                      //
// ======            =============                                      //
// C. Phillips       12 December 1995                                   //
//                                                                      //
// Input/Output                                                         //
// ============                                                         //
// The programs requires the input of an integer in the range 1 .. 7    //
//  that represents the level of curve to be drawn.                     //
// A hilbert curve of the specified level is drawn in a window.         //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <cmath>
using namespace std;

#include "Canvas.h"
#include "error.h"

//////////////////////////////////////////////////////////////////////////
//                          Type Definitions                            //
//////////////////////////////////////////////////////////////////////////

enum Direction {RIGHT, DOWN, LEFT, UP};

//////////////////////////////////////////////////////////////////////////
//                        Function Definitions                          //
//////////////////////////////////////////////////////////////////////////

void move(Direction direction, int h, Canvas& picture);
void hilbert(Direction r, Direction d, Direction l, Direction u ,
             int i, int h, Canvas& picture);

int main()
{
  const int XMAX = 256;
  const int YMAX = XMAX;
  int level;
  int h;
  Canvas picture;
  
  level = -1;
  do
  {
    cout << "What level of Hilbert curve do you want to draw (1 .. 7): ";
    cin >> level;
    if (cin.bad() || cin.eof())
    {
      error("An unexpected input error has occured");
    } 
    else if (cin.fail())          // Did an input error occur?
    {
      char c;
      
      cout << "That was not a legal number." << endl;
      level = -1;
      cin.clear();
      do
      {
        cin.get(c);
      } while (c != '\n');   // Skip the rest of the input line
    }
  } while ((level < 1) || (7 < level));
  
  picture.setSize(XMAX, YMAX);
  h = XMAX / int(pow(2.0, double(level)) + 0.5);
  picture.moveTo(h / 2, YMAX - h / 2);
  picture.penDown();
  cout << "Drawing the curve ..." << endl;
  hilbert(RIGHT, DOWN, LEFT, UP, level, h, picture); 
  cout << "Program complete" << endl;
  return 0;
}

void move(Direction direction, int h, Canvas& picture)
{
  switch (direction)
  {
    case RIGHT:
      {
        picture.moveRelative(h, 0);
      }
      break;
    case DOWN:
      {
        picture.moveRelative(0, -h);
      }
      break;
    case LEFT:
      {
        picture.moveRelative(-h, 0);
      }
      break;
    case UP:
      {
        picture.moveRelative(0, h);
      }
      break;
    default:
      {
        error("Unknown Direction");
      }
      break;
  }
  return;
} // end move()
  
void hilbert(Direction r, Direction d, Direction l, Direction u ,
             int i, int h, Canvas& picture)
{
  if (i>0)
  {
    hilbert(d, r, u, l, i - 1, h, picture);
    move(r, h, picture);
    hilbert(r, d, l, u, i - 1, h, picture);
    move(d, h, picture);
    hilbert(r, d, l, u, i - 1, h, picture);
    move(l, h, picture);
    hilbert(u, l, d, r, i - 1, h, picture);
  }
} // end hilbert()
