//////////////////////////////////////////////////////////////////////////
//                          DrawShapes.c                                //
//                                                                      //
// Purpose                                                              //
// =======                                                              //
// Implementation file for the shape-drawing functions.                 //
//                                                                      //
// Author            Creation Date                                      //
// ======            =============                                      //
// C. Phillips       12 December 1995                                   //
//////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <cmath>
using namespace std;

#include "Canvas.h"
#include "DrawShapes.h"

// Functions that are private to the implementation code.

bool   validTriangle(int side1, int side2, int side3);
double degreesToRadians(int angle);
void   convertToLocalPolar(int xOrigin, int yOrigin, double x, double y,
                           double& r, double& theta);
void   convertToLocalXY(double r, double theta, double& x, double& y);
void   convertToGlobalXY(int xOrigin, int yOrigin, double r, double theta,
                         double& x, double& y);
void   rotateCoordinates(int xOrigin, int yOrigin, double oldX, double oldY,
                         double& newX, double& newY, double rotationAngle);
void   convertCartesianToPolar(double x, double y, double&r, double& theta);
void   convertPolarToCartesian(double r, double theta, double&x, double& y);
double power2(double x);
int    roundToInt(double value);

// Some useful constants
#ifndef PI
const double PI    = 4.0 * atan(1.0);  // Sometimes already defined
#endif
const double TWOPI = 2.0 * PI;
const double PIBY2 = PI / 2.0;

                          //////////////////////////////
                          // Square Drawing Functions //
                          //////////////////////////////
                          
bool drawSquare(Canvas& picture)
{
  const int DEFAULT_LEN = 50;
  
  return drawSquare(picture, DEFAULT_LEN);
} // end drawSquare()

bool drawSquare(Canvas& picture, int sideLength)
{
  int xPos;
  int yPos;
  
  picture.getPosition(xPos, yPos);
  return drawSquare(picture, sideLength, xPos, yPos, 0);
} // end drawSquare()

bool drawSquare(Canvas& picture, int sideLength, int xPos, int yPos)
{
  return drawSquare(picture, sideLength, xPos, yPos, 0);
} // end drawSquare()
 
bool drawSquare(Canvas& picture, int sideLength, int xPos, int yPos,
                int rotationAngle)
{
  return drawSquare(picture, sideLength, xPos, yPos,
                    degreesToRadians(rotationAngle));
} // end drawSquare()

bool drawSquare(Canvas& picture, int sideLength, int xPos, int yPos,
                double rotationAngle)
{
  const int DISTS = roundToInt(double(sideLength) * sin(rotationAngle));
  const int DISTC = roundToInt(double(sideLength) * cos(rotationAngle));
    
  if (sideLength <= 0)  
  { 
    return false;                     // Terminate execution
  }
  
  picture.penUp();
  picture.moveTo(xPos, yPos);
  picture.penDown();
  picture.moveRelative(DISTC, DISTS);
  picture.moveRelative(-DISTS, DISTC);
  picture.moveRelative(-DISTC, -DISTS);
  picture.moveRelative(DISTS, -DISTC);

  picture.penUp();
  return true;
} // end drawSquare()
                
                         ////////////////////////////////
                         // Triangle Drawing Functions //
                         ////////////////////////////////
                         
bool drawTriangle(Canvas& picture)
{
  const int DEFAULT_LEN = 50;
  
  return drawTriangle(picture, DEFAULT_LEN, DEFAULT_LEN, DEFAULT_LEN);
} // end drawTriangle()

bool drawTriangle(Canvas& picture, int side1, int side2, int side3)
{
  int xPos;
  int yPos;
  
  picture.getPosition(xPos, yPos);
  return drawTriangle(picture, side1, side2, side3, xPos, yPos);
} // end drawTriangle()

bool drawTriangle(Canvas& picture, int side1, int side2, int side3,
                  int xPos, int yPos)
{
  return drawTriangle(picture, side1, side2, side3, xPos, yPos, 0);
} // end drawTriangle()

bool drawTriangle(Canvas& picture, int side1, int side2, int side3,
                  int xPos, int yPos, int rotationAngle)
{
  return drawTriangle(picture, side1, side2, side3, xPos, yPos, 
                      degreesToRadians(rotationAngle));
} // end drawTriangle()

bool drawTriangle(Canvas& picture, int side1, int side2, int side3,
                  int xPos, int yPos, double rotationAngle)
{
  double x = double(xPos);
  double y = double(yPos + side1);
  double actualX;
  double actualY;
  double side2Squared;
  double cosAngle3;
  double moveY;
  
  if (side1 <= 0 || side2 <= 0 || side3 <= 0 ||
      !validTriangle(side1, side2, side3)      )
  {
    return false;
  }
  picture.penUp();
  picture.moveTo(xPos, yPos);
  picture.penDown();
  
  rotateCoordinates(xPos, yPos, x, y, actualX, actualY, rotationAngle);  
  picture.moveTo(roundToInt(actualX), roundToInt(actualY));
  
  side2Squared  = power2(double(side2));
  cosAngle3     = (power2(double(side1)) + side2Squared - power2(double(side3)))
                   / double(2 * side1 * side2);                  
  moveY         = side2 * cosAngle3;
  x            += int(sqrt(side2Squared - power2(moveY)));
  y            -= moveY;
  rotateCoordinates(xPos, yPos, x, y, actualX, actualY, rotationAngle);  
  picture.moveTo(roundToInt(actualX), roundToInt(actualY));
  picture.moveTo(xPos, yPos);
  picture.penUp();
  return true;
} // end drawTriangle()

                          ///////////////////////////////
                          // Polygon Drawing Functions //
                          ///////////////////////////////

bool drawRegularPolygon(Canvas& picture)
{
  const int DEFAULT_SIDES = 4;
  
  return drawRegularPolygon(picture, DEFAULT_SIDES);
} // end drawRegularPolygon()                     

bool drawRegularPolygon(Canvas& picture, int numberOfSides)
{
  const int DEFAULT_LEN = 50;
  
  return drawRegularPolygon(picture, numberOfSides, DEFAULT_LEN);
} // end drawRegularPolygon()                     

bool drawRegularPolygon(Canvas& picture, int numberOfSides, int sideLength)
{
  int xPos;
  int yPos;
  
  picture.getPosition(xPos, yPos);  
  return drawRegularPolygon(picture, numberOfSides, sideLength,
                            xPos, yPos);
} // end drawRegularPolygon()                     

bool drawRegularPolygon(Canvas& picture, int numberOfSides, int sideLength, 
                        int xPos, int yPos)
{
  return drawRegularPolygon(picture, numberOfSides, sideLength, xPos, yPos, 0);
} // end drawRegularPolygon()                     

bool drawRegularPolygon(Canvas& picture, int numberOfSides, int sideLength, 
                        int xPos, int yPos, int rotationAngle)
{
  return drawRegularPolygon(picture, numberOfSides, sideLength, xPos, yPos, 
                            degreesToRadians(rotationAngle));
} // end drawRegularPolygon() 
                    
bool drawRegularPolygon(Canvas& picture, int numberOfSides, int sideLength, 
                        int xPos, int yPos, double rotationAngle)
{
  double xOrigin      = double(xPos);
  double yOrigin      = double(yPos);
  double angle        = TWOPI / double(numberOfSides);
  double currentAngle = 0;
  
  if (sideLength <= 0)
  { 
    return false;  // Terminate execution
  }
  
  picture.penUp();
  picture.moveTo(xPos, yPos);
  picture.penDown();
  
  rotationAngle = PIBY2 - rotationAngle;
  currentAngle  = 0;
  for (int side = 0; side < numberOfSides - 1; side += 1)
  {
    double localX;
    double localY;
    double globalX;
    double globalY;
    double x;
    double y; 
       
    convertPolarToCartesian(sideLength, currentAngle, localX, localY); 
    x = localX + xOrigin;
    y = localY + yOrigin;
    
    rotateCoordinates(xPos, yPos, x, y, globalX, globalY, rotationAngle); 
    picture.moveTo(roundToInt(globalX), roundToInt(globalY));
    xOrigin = x;
    yOrigin = y;
    currentAngle -= angle;
  }
  picture.moveTo(xPos, yPos);
  picture.penUp();
  return true;
} // end drawRegularPolygon()

                             ///////////////////////
                             // Private Functions //
                             ///////////////////////
                             
bool validTriangle(int side1, int side2, int side3)
{
  return (side1 < side2+side3) || (side2 < side1+side3) ||
         (side3 < side1 + side2);
} // end validTriangle()

double degreesToRadians(int angle)
{
  return (PI * angle / 180.0); // PI radians is equal to 180 degrees
} // end degreesToRadians()

void rotateCoordinates(int xOrigin, int yOrigin, double oldX, double oldY, 
                       double& newX, double& newY, double rotationAngle)
{
  double r;
  double theta;
  
  convertToLocalPolar(xOrigin, yOrigin, oldX, oldY, r, theta);  
  theta += rotationAngle;
  convertToGlobalXY(xOrigin, yOrigin, r, theta, newX, newY);
  return;
} // end rotateCoordinates()

void convertToLocalPolar(int xOrigin, int yOrigin, double x, double y,
                         double& r, double& theta)
{
  double shiftX = x - double(xOrigin);
  double shiftY = y - double(yOrigin);
  
  convertCartesianToPolar(shiftX, shiftY, r, theta);
  return;
} // end convertToLocalPolar()

void convertCartesianToPolar(double x, double y, double& r, double& theta)
{
  r     = sqrt(power2(x) + power2(y));
  theta = atan2(y, x);
  return;
} // end convertCartesianToPolar()

void convertPolarToCartesian(double r, double theta, double& x, double& y)
{
  x = r * cos(theta);
  y = r * sin(theta);
  return;
} // end convertPolarToCartesian()

void convertToGlobalXY(int xOrigin, int yOrigin, double r, double theta, 
                       double& x, double& y)
{ 
  convertPolarToCartesian(r, PIBY2 - theta, x, y);
  x = r * cos(theta) + xOrigin;
  y = r * sin(theta) + yOrigin;
  return;
} // end convertToGlobalXY()

double power2(double x)  // Return the square of the number given as a parameter
{
  return x * x;
} // end power2()

int roundToInt(double value)  // Round the double value to the nearest int value
{
  if (value < 0)
  {
    return int(value - 0.5); 
  }
  else
  {
    return int(value + 0.5);  
  }
} // end roundToInt()

