/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

import java.util.*;
import java.util.Collections;

import java.lang.Math;

class Facet
{
    private Vector<Vertex>       m_vertices;
    private Vector<DirectedEdge> m_edges;
    
    private double m_normal_x = 1.0;
    private double m_normal_y = 0.0;
    private double m_normal_z = 0.0;
    
    private double m_refdir_x = 0.0;
    private double m_refdir_y = 0.0;
    private double m_refdir_z = 0.0;
    
    private int m_iges_loop_de_seqno;
    private int m_iges_normal_de_seqno;
    private int m_iges_refdir_de_seqno;
    private int iges_surface_de_seqno;
    private int iges_face_de_seqno;

    private boolean m_bIntersects;
    private boolean m_bInternal;
    private boolean m_bExternal;
    
    private Facet   m_fext = null;
    private boolean m_bKeepExternal = false;

    private Vertex m_center = null;
    private double m_area;

    private String m_fid = null;

    public Facet (Vector f, Vector v)
    {
	m_vertices = new Vector<Vertex> (f.size ());
	m_edges    = new Vector<DirectedEdge> (f.size ());

	for(int i=0; i<f.size(); i++)
	    {
		Integer VID = (Integer) f.elementAt (i);
		int vid = VID.intValue ();

		for(int j=0; j<v.size(); j++)
		    {
			Vertex vx = (Vertex)v.elementAt(j);
			if (vx.vid () == vid)
			    {
				m_vertices.addElement (vx);
				break;
			    }
			
		    }
	    }

    }
    public Facet (Vector<EdgeKey> edgekey, PlaneCompare pc, TreeMap<EdgeKey,Edge> edge)
    {   
	m_vertices = new Vector<Vertex> (edgekey.size ());
	m_edges    = new Vector<DirectedEdge> (edgekey.size ());

	EdgeKey k1 = edgekey.elementAt(0);

	m_vertices.addElement (k1.v1 ());
	m_vertices.addElement (k1.v2 ());

	int length = edgekey.size();
	
	Vertex v11 = k1.v1();
	Vertex v12 = k1.v2();
	
	for (int i = 1; i < length - 1; i++)
	    {
		for (int j = 1; j < length; j++)
		    {
			EdgeKey k2 = edgekey.elementAt(j);

			Vertex v21 = k2.v1();
			Vertex v22 = k2.v2();

			if ((v12 == v21) && (v11 != v22))
			    {
				m_vertices.addElement (v22);
				v11 = v21;
				v12 = v22;
				break;
			    }
			if ((v12 == v22) && (v11 != v21))
			    {
				m_vertices.addElement (v21);
				v11 = v22;
				v12 = v21;
				break;
			    }
		    }
	    }
	
	m_normal_x = pc.normal_x ();
	m_normal_y = pc.normal_y ();
	m_normal_z = pc.normal_z ();
	
	Vertex va = m_vertices.elementAt (0);
	Vertex vb = m_vertices.elementAt (1);

	double d1x = vb.xd () - va.xd ();
	double d1y = vb.yd () - va.yd ();
	double d1z = vb.zd () - va.zd ();
	
	va = vb;
	vb = m_vertices.elementAt (2);
	
	double d2x = vb.xd () - va.xd ();
	double d2y = vb.yd () - va.yd ();
	double d2z = vb.zd () - va.zd ();

	double dnx = d1y * d2z - d1z * d2y;
	double dny = d1z * d2x - d1x * d2z;
	double dnz = d1x * d2y - d1y * d2x;
	
	if (dnx * m_normal_x + dny * m_normal_y + dnz * m_normal_z > 0)
	    {
		Collections.reverse (m_vertices); // vertices are in wrong order
	    }
    }
    public Facet (Vector<Vertex> v, double normal_x, double normal_y, double normal_z)
    {
	m_vertices = new Vector<Vertex> (v.size ());
	m_edges    = new Vector<DirectedEdge> (v.size ());

	for (int j = 0; j < v.size (); j++)
	    {
		m_vertices.addElement (v.elementAt (j));
	    }

	m_normal_x = normal_x;
	m_normal_y = normal_y;
	m_normal_z = normal_z;
    }
    
    public void map_edges (TreeMap<EdgeKey,Edge> edge)
    {
	for( int i = 0; i < (m_vertices.size()); i++ )
	    {
		Vertex v1;
		Vertex v2;

		if (i != (m_vertices.size () - 1))
		    {
			v1 = m_vertices.elementAt (i);
			v2 = m_vertices.elementAt (i+1);
		    }
		else
		    {
			v1 = m_vertices.elementAt (i);
			v2 = m_vertices.elementAt (0);
		    }

		EdgeKey k1 = new EdgeKey(v1, v2);
		Edge    e1 = new Edge(v1, v2);

		edge.put (k1, e1);
       	    }
    }

    public void make_facet_edges (TreeMap<EdgeKey,Edge> edge, boolean bCalculateNormal)
    {
	m_edges.clear ();

	for (int i = 0; i < (m_vertices.size()); i++ )
	    {
		Vertex v1;
		Vertex v2;
		
		if (i != (m_vertices.size () - 1))
		    {
			v1 = m_vertices.elementAt (i);
			v2 = m_vertices.elementAt (i+1);
		    }
		else
		    {
			v1 = m_vertices.elementAt (i);
			v2 = m_vertices.elementAt (0);
		    }
		
		EdgeKey k1 = new EdgeKey(v1, v2);

		Edge e1 =  edge.get(k1);

// if (e1 == null) { System.out.print (String.format ("oops! - no edge (%d,%d)?",v1.vid(),v2.vid())); System.out.flush (); }

		m_edges.addElement (new DirectedEdge (e1, k1.aligned ()));
	    }

	if (bCalculateNormal)
	    {
		int edge_count = m_edges.size ();

		double nx = 0;
		double ny = 0;
		double nz = 0;

		boolean bFirst = true;

		for (int i1 = 0; i1 < (edge_count - 1); i1++)
		    {
			DirectedEdge e1 = m_edges.elementAt (i1);

			Vertex v11 = e1.v1 ();
			Vertex v12 = e1.v2 ();

			double d1x = v12.xd () - v11.xd ();
			double d1y = v12.yd () - v11.yd ();
			double d1z = v12.zd () - v11.zd ();

			for (int i2 = i1 + 1; i2 < edge_count; i2++)
			    {
				DirectedEdge e2 = m_edges.elementAt (i2);

				Vertex v21 = e2.v1 ();
				Vertex v22 = e2.v2 ();

				double d2x = v22.xd () - v21.xd ();
				double d2y = v22.yd () - v21.yd ();
				double d2z = v22.zd () - v21.zd ();

				double dnx = d1y * d2z - d1z * d2y;
				double dny = d1z * d2x - d1x * d2z;
				double dnz = d1x * d2y - d1y * d2x;

				if (bFirst)
				    {
					m_normal_x = dnx;
					m_normal_y = dny;
					m_normal_z = dnz;
					bFirst = false;
				    }

				double sense = 1;

				if (dnx * m_normal_x + dny * m_normal_y + dnz * m_normal_z > 0)
				    {
					sense = -1;
				    }

				nx += dnx * sense;
				ny += dny * sense;
				nz += dnz * sense;
			    }
		    }
		double length = Math.sqrt (nx * nx + ny * ny + nz * nz);

		m_normal_x = nx / length;
		m_normal_y = ny / length;
		m_normal_z = nz / length;
	    }

	DirectedEdge e0 = m_edges.elementAt (0);

	Vertex v01 = e0.v1 ();
	Vertex v02 = e0.v2 ();

	m_refdir_x = v02.xd () - v01.xd ();
	m_refdir_y = v02.yd () - v01.yd ();
	m_refdir_z = v02.zd () - v01.zd ();

	double length = Math.sqrt (m_refdir_x * m_refdir_x + 
				   m_refdir_y * m_refdir_y + 
				   m_refdir_z * m_refdir_z);

	m_refdir_x /= length;
	m_refdir_y /= length;
	m_refdir_z /= length;
    }

    public double refdir_x ()
    {
	return m_refdir_x;
    }
    public double refdir_y ()
    {
	return m_refdir_y;
    }
    public double refdir_z ()
    {
	return m_refdir_z;
    }

    public float normal_xf ()
    {
	return (float)m_normal_x;
    }
    public float normal_yf ()
    {
	return (float)m_normal_y;
    }
    public float normal_zf ()
    {
	return (float)m_normal_z;
    }

    public double normal_xd ()
    {
	return m_normal_x;
    }
    public double normal_yd ()
    {
	return m_normal_y;
    }
    public double normal_zd ()
    {
	return m_normal_z;
    }

    public final Vector<Vertex> vertices ()
    {
	return m_vertices;
    }
    public final Vector<DirectedEdge> edges ()
    {
	return m_edges;
    }
    
    public void set_iges_loop_de_seqno (int i)
    {
	m_iges_loop_de_seqno = i;
    }
    public int get_iges_loop_de_seqno()
    {
	return m_iges_loop_de_seqno;
    }

    public void set_iges_normal_de_seqno (int i)
    {
	m_iges_normal_de_seqno = i;
    }
    public int get_iges_normal_de_seqno ()
    {
	return m_iges_normal_de_seqno;
    }

    public void set_iges_refdir_de_seqno (int i)
    {
	m_iges_refdir_de_seqno = i;
    }
    public int get_iges_refdir_de_seqno ()
    {
	return m_iges_refdir_de_seqno;
    }

    public void set_iges_surface_de_seqno (int i)
    {
	iges_surface_de_seqno = i;
    }
    public int get_iges_surface_de_seqno ()
    {
	return iges_surface_de_seqno;
    }

    public int get_iges_point_de_seqno ()
    {
	Vertex v = m_vertices.elementAt(0);
	return v.get_iges_point_de_seqno ();
    }
    public void set_iges_face_de_seqno( int i)
    {
	iges_face_de_seqno = i;
    }
    public int get_iges_face_de_seqno()
    {
	return iges_face_de_seqno;
    }
    
    public void set_internal (boolean bInternal)
    {
	m_bInternal = bInternal;
    } 
    public boolean get_internal ()
    {
	return m_bInternal;
    }
    
    public void set_external (boolean bExternal)
    {
	m_bExternal = bExternal;
    }
    public boolean get_external ()
    {
	return m_bExternal;
    }
    
    public void set_intersects (boolean bIntersects)
    {
	m_bIntersects = bIntersects;
    }
    public boolean get_intersects ()
    {
	return m_bIntersects; 
    }
    public void check_edges ()
    {
	set_intersects(false);
	set_internal(true);
	set_external(true);

	int length = m_edges.size();

	for (int j = 0; j<length; j++)
	    {
		DirectedEdge de = m_edges.elementAt(j);
		Edge e = de.edge();
		
		if (e.get_intersects())
		    {
			set_intersects(true);
			set_internal(false);
			set_external(false);
			break;
		    }
		if (!e.get_external())
		    {
			set_external(false);
		    }
		if (!e.get_internal())
		    {
			set_internal(false);
		    }
	    }
	if (!get_internal() && !get_external())
	    {
		set_intersects(true);
	    }
    }
    public EdgeKey crop_to_plane (double offset, PlaneCompare pc, boolean bKeepExternal)
    {
	Vector<Vertex> vext = null;

	if (bKeepExternal)
	    vext = new Vector<Vertex> ();

	Vertex v_entry = null;
	Vertex v_exit  = null;

	m_vertices.clear ();

	int length = m_edges.size ();

	for (int i = 0; i < length; i++)
	    {
		DirectedEdge de = m_edges.elementAt(i);
		Edge e = de.edge();

		Vertex v1 = de.v1 ();
		Vertex v2 = de.v2 ();
	
		if (!pc.is_external (offset, v1))
		    {
			m_vertices.addElement (v1);
			
			if (e.get_intersects())
			    {
				v_exit = e.get_intersection ();
				m_vertices.addElement (v_exit);
				if (bKeepExternal)
				    vext.addElement (v_exit);
			    }
			else if (pc.is_external (offset, v2))
			    {
				v_exit = v2;
				m_vertices.addElement (v_exit);
				if (bKeepExternal)
				    vext.addElement (v_exit);
			    }
		    }
		if (!pc.is_internal (offset, v1))
		    {
			if (bKeepExternal)
			    vext.addElement (v1);

			if (e.get_intersects())
			    {
				v_entry = e.get_intersection ();
				m_vertices.addElement (v_entry);
				if (bKeepExternal)
				    vext.addElement (v_entry);
			    }
			else if (pc.is_internal (offset, v2))
			    {
				v_entry = v2;
				m_vertices.addElement (v_entry);
				if (bKeepExternal)
				    vext.addElement (v_entry);
			    }
		    }
	    }

	if (bKeepExternal)
	    m_fext = new Facet (vext, m_normal_x, m_normal_y, m_normal_z);
	else
	    m_fext = null;

	return new EdgeKey (v_entry, v_exit);
    }
    public Facet get_external_facet () // only use once! (removes internal facet)
    {
	return m_fext;
    }

    public EdgeKey edge_on_plane (double offset, PlaneCompare pc) // method may return null
    {
	Vertex v_entry = null;
	Vertex v_exit  = null;

	int length = m_edges.size ();
	
	for (int i = 0; i < length; i++)
	    {
		DirectedEdge E = m_edges.elementAt(i);

		Vertex v1 = E.v1();
		Vertex v2 = E.v2();
		
		if  (pc.is_external (offset, v1) && pc.is_external (offset, v2))
		    {
			v_entry = v1;
			v_exit  = v2;
			break;
		    }
	    }

	if ((v_entry == null) && (v_exit == null))
	    {
		return null;
	    }
	return new EdgeKey (v_entry, v_exit);
    }
    public void save (RegionLoader rl)
    {
	rl.write(String.format("   <Facet vertices=\"%d\">",m_vertices.size()));
	for (int j = 0; j<m_vertices.size(); j++)
	    {
		Vertex v = m_vertices.elementAt(j);
		rl.write(String.format("%d\t",v.vid()));
	    }
	rl.write("</Facet>\n");

	rl.write(String.format("   <Edges count=\"%d\">",m_edges.size()));
	for (int j = 0; j<m_edges.size(); j++)
	    {
		DirectedEdge de = m_edges.elementAt(j);
		rl.write(String.format("(%d,%d)\t",de.v1().vid(),de.v2().vid()));
	    }
	rl.write("</Edges>\n");
    }

    public void zero_edge_counts ()
    {
	for (int i = 0; i < m_edges.size (); i++)
	    {
		DirectedEdge de = m_edges.elementAt (i);
		Edge e = de.edge ();
		e.count_clear ();
	    }
    }
    public void mark_edges ()
    {
	for (int i = 0; i < m_edges.size (); i++)
	    {
		DirectedEdge de = m_edges.elementAt (i);
		Edge e = de.edge ();
		e.count_increment ();
	    }
    }
    public void check_edge_count ()
    {
	for (int i = 0; i < m_edges.size (); i++)
	    {
		DirectedEdge de = m_edges.elementAt (i);
		Edge e = de.edge ();

		int count = e.count ();
		if (count != 2)
		    {
			System.out.print (String.format ("#facets = %d!\n", count));
		    }
	    }
    }

    public void calculate_center_and_area ()
    {
	double x = 0;
	double y = 0;
	double z = 0;

	int count = m_vertices.size ();

	for (int j = 0; j < count; j++)
	    {
		Vertex v = m_vertices.elementAt (j);

		x += v.xd ();
		y += v.yd ();
		z += v.zd ();
	    }
	m_center = new Vertex (x / count, y / count, z / count);

	m_area = 0;

	for (int j = 0; j < m_edges.size (); j++)
	    {
		DirectedEdge de = m_edges.elementAt (j);

		Vertex v1 = de.v1 ();
		Vertex v2 = de.v2 ();

		double edx = v2.xd () - v1.xd ();
		double edy = v2.yd () - v1.yd ();
		double edz = v2.zd () - v1.zd ();

		double cdx = m_center.xd () - v1.xd ();
		double cdy = m_center.yd () - v1.yd ();
		double cdz = m_center.zd () - v1.zd ();

		double Xdx = edy * cdz - edz * cdy;
		double Xdy = edz * cdx - edx * cdz;
		double Xdz = edx * cdy - edy * cdx;

		m_area += Math.sqrt (Xdx * Xdx + Xdy * Xdy + Xdz * Xdz) / 2;
	    }
    }
    public final Vertex center ()
    {
	return m_center;
    }
    public double area ()
    {
	return m_area;
    }

    public String make_ID ()
    {
	m_fid = null;

	Vector<Integer> f = new Vector<Integer>(m_vertices.size ());

	for (int j = 0; j < m_vertices.size (); j++)
	    {
		Vertex v = m_vertices.elementAt (j);
		f.addElement (v.vid ());
	    }

	Collections.sort (f);

	Enumeration<Integer> E = f.elements ();

	while (E.hasMoreElements ())
	    {
		Integer I = E.nextElement ();
		if (m_fid == null)
		    m_fid = I.toString ();
		else
		    m_fid = String.format ("%s:%s", m_fid, I.toString ());
	    }
	return m_fid;
    }
    public String get_ID ()
    {
	return m_fid;
    }
}
