/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

import java.util.*;
import java.lang.*;
import java.io.*;

class IGES
{
    private int  m_line_number;
    
    private boolean fcloseflag;
	    
    private IGES_Section m_IS;
    	    
    private String m_start;
	    
    private Parameter m_global[] = new Parameter[24];
    	    
    private Vector<DirectoryEntry> m_directory_entry;

    private Map<IGES_Section, Integer> m_counts = new EnumMap<IGES_Section, Integer>(IGES_Section.class);

    String m_line_buffer;
	    
    public IGES ()
    {
	
	m_directory_entry = new Vector<DirectoryEntry> ();
	m_line_buffer = new String();
	m_line_number = 1;
	m_IS = IGES_Section.is_Start;
	m_start = new String("");
	fcloseflag = false;
	for (int i = 0; i < 24;i++)
	    m_global[i] = new Parameter();	
	build_global ();
	
    }
	    
    public IGES(String start)
    {
	m_directory_entry = new Vector<DirectoryEntry>();
	m_line_buffer = new String();
	m_line_number = 1;
	m_IS = IGES_Section.is_Start;
	m_start = new String(start);
	fcloseflag = false;
	for (int i = 0; i < 24;i++)
	    m_global[i] = new Parameter();
	
	build_global ();
    }

    public void new_section (IGES_Section is)
    {
	m_IS = is;
	m_line_number = 0;
	m_counts.put (m_IS, m_line_number++);
    }
    public DirectoryEntry append_directory_entry (int entity_number)
    {
	return append_directory_entry (entity_number, 0);
    }
    public DirectoryEntry append_directory_entry (int entity_number, int form_number)
    {   
	int sequence_number = 1 + 2 * m_directory_entry.size ();
	DirectoryEntry de = new DirectoryEntry (entity_number, form_number, sequence_number);
	m_directory_entry.addElement (de);
	return de;
    }
    public Parameter global_parameter (GlobalParameter gp) { return m_global[gp.getvalue()]; }

    public void print_line (FileOutputStream fout) {
	
	while (m_line_buffer.length () < 72)
	    m_line_buffer = m_line_buffer.concat (" ");
	
	m_line_buffer = m_line_buffer.concat (String.format ("%c%7d\r\n", m_IS.getvalue(), m_line_number));
	
	byte buf[] = m_line_buffer.getBytes();
	try {
	    fout.write(buf);
	} catch ( Exception e){}
	
	// System.out.println(m_line_buffer);
	m_counts.put (m_IS, m_line_number++);
	    
    }
    
    void clear_line ()
    {
	m_line_buffer = "";
    }
    
    void Write (String filename) throws IOException
    {
	File f = new File(filename);
	FileOutputStream fout;
	fout = new FileOutputStream ( f );
	
	new_section (IGES_Section.is_Start);

	clear_line ();
		
	if (m_start.length () > 72)
	    m_line_buffer = m_start.substring (0,71);
	else
	    m_line_buffer = m_start;
		
	print_line (fout);

	new_section (IGES_Section.is_Global);
		
	m_line_buffer = ",,";
	int buffer_offset = 2;
		
	int global_count = 24;

	for (int i = 0; i < global_count; i++)
	    {
		String delim = ((i + 1) == global_count) ?(";") :(",");

		Parameter GP = m_global[i];
			
		int length = GP.parameter().length ();
		int offset = 0;
			
		if (buffer_offset + length < 72)
		    {
			m_line_buffer = m_line_buffer.concat (GP.parameter ());
			buffer_offset += length;
			m_line_buffer = m_line_buffer.concat (delim);
			buffer_offset++;
			continue;
		    }
		if (GP.bNumeric ())
		    {
			print_line (fout);
			clear_line ();
				
			buffer_offset = 0;
			m_line_buffer = m_line_buffer.concat (GP.parameter ());
			buffer_offset += length;
			m_line_buffer = m_line_buffer.concat (delim);
			buffer_offset++;
			continue;
		    }
		while (offset < length)
		    {
			if ((length - offset) <= (72 - buffer_offset))
			    {
				m_line_buffer = m_line_buffer.concat (GP.parameter().substring (offset, length));
				buffer_offset += length - offset;
				offset += length - offset;
			    }
			else
			    {
				m_line_buffer = m_line_buffer.concat (GP.parameter().substring (offset, offset + 72 - buffer_offset));
				offset += 72 - buffer_offset;
				buffer_offset += 72 - buffer_offset;
			    }
			if (buffer_offset == 72)
			    {
				print_line (fout);
				clear_line ();
				
				buffer_offset = 0;
			    }
		    }
		m_line_buffer = m_line_buffer.concat (delim);
		buffer_offset++;
	    }
	if (buffer_offset > 0)
	    print_line (fout);

	build_sequence ();
		
	new_section (IGES_Section.is_DirectoryEntry);
		
	int de_count = m_directory_entry.size ();
		
	for (int de = 0; de < de_count; de++)
	    {
		DirectoryEntry DE = (DirectoryEntry) m_directory_entry.elementAt (de);
			
		m_line_buffer = DE.record_line_1 ();
		print_line (fout);
			
		m_line_buffer = DE.record_line_2 ();
		print_line (fout);
	    }
		
	new_section (IGES_Section.is_ParameterData);
		
	for (int de = 0; de < de_count; de++)
	    {
		DirectoryEntry DE = (DirectoryEntry) m_directory_entry.elementAt (de);
			
		ParameterData PD = DE.data ();
			
		Vector  sequence = PD.sequence ();
			
		int line_count = sequence.size ();
			
		for (int l = 0; l < line_count; l++)
		    {
			m_line_buffer = (String) sequence.elementAt (l);
			print_line (fout);
		    }
	    }
		
	new_section (IGES_Section.is_Terminate);

	Integer CS = (Integer) m_counts.get (IGES_Section.is_Start);
	Integer CG = (Integer) m_counts.get (IGES_Section.is_Global);
	Integer CD = (Integer) m_counts.get (IGES_Section.is_DirectoryEntry);
	Integer CP = (Integer) m_counts.get (IGES_Section.is_ParameterData);
		
	m_line_buffer = String.format( "%c%7d%c%7d%c%7d%c%7d",
	       IGES_Section.is_Start.getvalue(),		CS.intValue (),
	       IGES_Section.is_Global.getvalue(),		CG.intValue (),
	       IGES_Section.is_DirectoryEntry.getvalue(),	CD.intValue (),
	       IGES_Section.is_ParameterData.getvalue(),	CP.intValue ());
		
	print_line (fout);
	try {
		fout.close();
	    }catch(Exception e){}
    }

    public void set_global_name (String name)
    {
       	m_global[GlobalParameter.gp_name_sender.getvalue()  ].set_HS (name); // Product identification from sending system
	m_global[GlobalParameter.gp_name_receiver.getvalue()].set_HS (name);
    }

    public void set_global_filename (String name)
    {
	m_global[GlobalParameter.gp_name_file.getvalue()].set_HS (name);
    }

    public void set_global_app_version (String application, String version)
    {
	m_global[GlobalParameter.gp_application_name.getvalue()   ].set_HS (application); // Native System ID
	m_global[GlobalParameter.gp_application_version.getvalue()].set_HS (version);     // Preprocessor version
    }
    public void set_global_units (ModelUnits mu)
    {
	set_global_units (mu, 1);
    }
    	 void set_global_units (ModelUnits mu, double scale)
    {
	switch (mu)
		{
		case mu_Inches:
		    m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (1);
		    m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("IN");
			break;
		case mu_Millimeters:
		    m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (2);    // Units flag
		    m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("MM"); // Units Name
		    break;

// When field 14 is 3, the string naming the desired unit shall conform to [MIL12] or [IEEE260].

		case mu_Feet:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (4);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("FT");
			break;
		case mu_Miles:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (5);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("MI");
			break;
		case mu_Meters:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (6);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("M");
			break;
		case mu_Kilometers:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (7);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("KM");
			break;
		case mu_Mils:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (8);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("MIL");
			break;
		case mu_Microns:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (9);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("UM");
			break;
		case mu_Centimeters:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (10);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("CM");
			break;
		case mu_Microinches:
			m_global[GlobalParameter.gp_units_flag.getvalue()].set_ld (11);
			m_global[GlobalParameter.gp_units_name.getvalue()].set_HS ("UIN");
			break;
		}

	// Model space scale
	m_global[GlobalParameter.gp_space_scale.getvalue()].set_lG (scale);

    }
    public void set_global_line_weight (int gradations, double maximum_weight)
    {
	// Maximum number of line weight gradations. Refer to the Directory Entry
	// Parameter 12.
	m_global[GlobalParameter.gp_maximum_line_weight_gradations.getvalue()].set_ld (gradations);

	// Width of maximum line weight in units. Refer to the Directory Entry Parameter
	// 12 for use of this parameter.
	m_global[GlobalParameter.gp_maximum_line_weight.getvalue()].set_lG (maximum_weight);

    }
       public void set_global_canvas (double resolution, double maximum_coordinate)
    {
	// Minimum user-intended resolution or granularity of the model in units specified
	// by Parameter 14.
	m_global[GlobalParameter.gp_resolution.getvalue()].set_lG (resolution);

	// Approximate maximum coordinate value occurring in the model in units
	// specified by Parameter 14.
	m_global[GlobalParameter.gp_maximum_coordinate.getvalue()].set_lG (maximum_coordinate);
    }
      void set_global_author (String author_name, String author_organization)
    {
	// Name of author
	m_global[GlobalParameter.gp_author_name.getvalue()].set_HS (author_name);

	// Author's organization
	m_global[GlobalParameter.gp_author_organization.getvalue()].set_HS (author_organization);
    }

    
    void build_global () {
	set_global_name ( "Part1" );

	set_global_filename ( "No Iges File Name specified" );

	// FIXME - Check!!
	set_global_app_version ("AutoDesk Inventor R10", "AutoDesk Inventor Iges Exporter R10");

	// Number of binary bits for integer representation
	m_global[GlobalParameter.gp_integer_bits.getvalue()].set_ld (32);

	// Maximum power often representable in a single-precision floating point number on the sending system
	m_global[GlobalParameter.gp_float_power.getvalue()].set_ld (38);

	// Number of significant digits in a single-precision floating point number on the sending system
	m_global[GlobalParameter.gp_float_digits.getvalue()].set_ld (6);

	// Maximum power of ten representable in a double-precision floating point number on the sending system
	m_global[GlobalParameter.gp_double_power.getvalue()].set_ld (99);

	// Number of significant digits in a double-precision floating point number on the sending system
	m_global[GlobalParameter.gp_double_digits.getvalue()].set_ld (15);

	set_global_units (ModelUnits.mu_Millimeters);

	set_global_line_weight (1, 0.08);

	// Date and time of exchange file generation
	//   15HYYYYMMDD.HHNNSS or 13HYYMMDD.HHNNSS
	//     where:
	//     YYYY or YY is 4 or 2 digit year   HH is hour (00-23)
	//     MM is month (01-12)               NN is minute (00-59)
	//     DD is day (01-31)                 SS is second (00-59)
	// m_global[gp_generation_date].set (std::string (""), false);

	set_global_canvas (0.01, 10000);

	set_global_author (("No Author specified"),  ("No Organization specified"));

	// Flag value corresponding to the version of the Specification to which this file complies.
	m_global[GlobalParameter.gp_specification_version.getvalue()].set_ld (11);

	// Flag value corresponding to the drafting standard to which this file complies, if any.
	m_global[GlobalParameter.gp_specification_draft.getvalue()].set_ld (0);

	// 	Date and time the model was created or last modified, in same format as field 18.
	// m_global[gp_modification_date].set (std::string (""), false);

	// Descriptor indicating application protocol, application subset, Mil-specification,
	// or user-defined protocol or subset, if any.
	m_global[GlobalParameter.gp_application_protocol.getvalue()].set_HS ( ("No Application Protocol specified"));
	
    } // set default values of global parameters
    
    void build_sequence () 
    {
        int de_count = m_directory_entry.size ();
	int pd_seqno = 1;
	
	for (int de = 0; de < de_count; de++)
	    {
		DirectoryEntry DE = (DirectoryEntry)m_directory_entry.elementAt(de);
		
		ParameterData PD = DE.data ();
		
		pd_seqno = PD.build_sequence (DE.sequence_number (), pd_seqno); // method updates pd_seqno
		DE.build_sequence ();
	    }
	
    }
         
    int add_point (double x, double y, double z)
    {
	DirectoryEntry DE = append_directory_entry (116);
	
	ParameterData  PD = DE.data ();
	
	PD.append_lG (x);
	PD.append_lG (y);
	PD.append_lG (z);
	
	PD.append_ld (0); // symbol (none)
	
	return DE.sequence_number ();
    }

    void add_points (RegionLoader rl)
    {
	int N = rl.vertex_vector().size();
		
	for (int n = 0; n < N; n++)
	    {
		Vertex v = (Vertex)rl.vertex_vector().elementAt(n);
		v.set_iges_point_de_seqno (add_point (v.xd(), v.yd(), v.zd()));
	    }
    }

      int add_line (double x1, double y1, double z1, double x2, double y2, double z2)
    {
	DirectoryEntry DE = append_directory_entry (110);
	
	ParameterData PD = DE.data ();
	
	PD.append_lG (x1);
	PD.append_lG (y1);
	PD.append_lG (z1);
	
	PD.append_lG (x2);
	PD.append_lG (y2);
	PD.append_lG (z2);

	return DE.sequence_number ();
    }

    void add_lines (RegionLoader rl)
    {
	TreeMap<EdgeKey,Edge> map = rl.edge_map();
	Collection<Edge> collection = map.values ();
	Iterator<Edge> i = collection.iterator ();
	
	while(i.hasNext())
	    {
		Edge e = i.next();

		Vertex v1 = e.v1();
		Vertex v2 = e.v2();

		e.set_iges_line_de_seqno (add_line (v1.xd(), v1.yd(), v1.zd(), v2.xd(), v2.yd(), v2.zd()));
	    }
    }
    
    int add_vertex_list (RegionLoader rl)
    {
	DirectoryEntry DE = append_directory_entry (502, 1);
	
	ParameterData PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);

	int N = rl.vertex_vector().size();
	
	PD.append_ld (N);
	
	for (int n = 0; n < N; n++)
	    {
		Vertex v = (Vertex)rl.vertex_vector().elementAt(n);	
		PD.append_lG (v.xd());
		PD.append_lG (v.yd());
		PD.append_lG (v.zd());
		v.set_iges_index(n+1);
	    }
	
	return DE.sequence_number ();
    }
    
    int add_edge_list (RegionLoader rl)
    {
	DirectoryEntry DE = append_directory_entry (504, 1);

	ParameterData  PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);
	DE.set_hierarchy (StatusHierarchy.sh_GlobalDefer);
	
	TreeMap<EdgeKey,Edge> map = rl.edge_map();
	Collection<Edge> collection = map.values ();
	Iterator<Edge> i = collection.iterator ();

	PD.append_ld(collection.size());
	int j = 1;
	while(i.hasNext())
	    {
		Edge e = i.next();
		e.set_iges_edge_index(j);
		j++;
		PD.append_ld(e.get_iges_line_de_seqno());
		PD.append_ld(rl. get_iges_vertex_list_de_seqno());

		Vertex v1 = e.v1();
		PD.append_ld(v1.get_iges_index());
		
		PD.append_ld(rl. get_iges_vertex_list_de_seqno());
		Vertex v2 = e.v2();
		PD.append_ld(v2.get_iges_index());
				
	    }
	
	return DE.sequence_number ();
    }
    
    int add_loop (Facet f,RegionLoader rl)
    {
	DirectoryEntry DE = append_directory_entry (508, 1);
	
	ParameterData PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);
	
	PD.append_ld (f.edges().size());
	
	for(int i = 0; i<f.edges().size(); i++)
	    {
		PD.append_ld (0);
		DirectedEdge e1 = f.edges().elementAt (i);
		PD.append_ld (rl.get_iges_edge_list_de_seqno());
		PD.append_ld (e1.get_iges_edge_index());
		if (e1.aligned() == true)
		    {
			PD.append_ld (1);
		    }
		else
		    {
			PD.append_ld (0);
		    }
		PD.append_ld (0);
		// PD.append_ld (0); // only if above is non-zero
		// PD.append_ld (0);
	    }
	
	return DE.sequence_number ();
    }

    public int add_direction (double x, double y, double z)
    {
	DirectoryEntry DE = append_directory_entry (123, 1);
	
	ParameterData PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);
	DE.set_hierarchy (StatusHierarchy.sh_GlobalDefer);
	DE.set_entity (StatusEntity.se_Definition);
		
	PD.append_lG (x);
	PD.append_lG (y);
	PD.append_lG (z);

	return DE.sequence_number ();
	
    }
    public int add_surface (int point_de_seqno,int  normal_de_seqno,int  refdir_de_seqno)
    {
	DirectoryEntry  DE = append_directory_entry (190, 1);
	
	ParameterData  PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);

	PD.append_ld (point_de_seqno);
	PD.append_ld (normal_de_seqno);
	PD.append_ld (refdir_de_seqno);
	
	return DE.sequence_number ();
    }
    
    public int add_face (int surface_de_seqno, int loop_de_seqno)
    {
	DirectoryEntry  DE = append_directory_entry (510, 1);
	
	ParameterData  PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);
	DE.set_hierarchy (StatusHierarchy.sh_GlobalDefer);

	PD.append_ld (surface_de_seqno);
	PD.append_ld (1);
	PD.append_ld (1);
	PD.append_ld (loop_de_seqno);

	return DE.sequence_number ();
    }

    public void add_faces (RegionLoader rl)
    {
	int Nj = rl.region_vector().size ();
	for (int j = 0; j < Nj; j++)
	    {
		Region r = rl.region_vector().elementAt (j);
		int Nk = r.region_facet_v().size ();
		for (int k = 0; k < Nk; k++)
		    {		
			Facet f = r.region_facet_v().elementAt (k);

			int point_de_seqno = f.get_iges_point_de_seqno (); // ask for point_de_seqno of first vertex
			
			int normal_de_seqno = add_direction (f.normal_xd (), f.normal_yd (), f.normal_zd ());
			f.set_iges_normal_de_seqno (normal_de_seqno);

			int refdir_de_seqno = add_direction (f.refdir_x (), f.refdir_y (), f.refdir_z ());
			f.set_iges_refdir_de_seqno (refdir_de_seqno);

			int surface_de_seqno = add_surface (point_de_seqno, normal_de_seqno, refdir_de_seqno);
			f.set_iges_surface_de_seqno (surface_de_seqno);

			int loop_de_seqno = add_loop (f, rl);
			f.set_iges_loop_de_seqno (loop_de_seqno);

			int face_de_seqno = add_face (surface_de_seqno, loop_de_seqno);
			f.set_iges_face_de_seqno (face_de_seqno);
		    }
	    }
    }
    
    public int add_shell (Region r, RegionLoader rl)
    {
	DirectoryEntry  DE = append_directory_entry (514, 1);
	
	ParameterData  PD = DE.data ();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);

	int Nk = r.region_facet_v().size ();

	PD.append_ld (Nk);

	for (int k = 0; k < Nk; k++)
	    {
		Facet f = r.region_facet_v().elementAt (k);

		PD.append_ld (f.get_iges_face_de_seqno ());
		PD.append_ld (1);
	    }

	return DE.sequence_number ();
    }
    public int add_color (Region r)
    {
	DirectoryEntry  DE = append_directory_entry (314, 0);
	
	ParameterData  PD = DE.data ();

	DE.set_entity (StatusEntity.se_Definition);
	
	PD.append_lG (((double) r.color_r ()) * 100);
	PD.append_lG (((double) r.color_g ()) * 100);
	PD.append_lG (((double) r.color_b ()) * 100);
	PD.append_HS (r.color_name ());

	return DE.sequence_number ();
    }

    public int add_manifold_solid (Region r, RegionLoader rl)
    {
	int shell_de_seqno = add_shell (r, rl);
	int color_de_seqno = add_color (r);

	DirectoryEntry  DE = append_directory_entry (186, 0);

	DE.set_color (-color_de_seqno);

	ParameterData  PD = DE.data ();

	PD.append_ld (shell_de_seqno);
	PD.append_ld (1);
	PD.append_ld (0);

	return DE.sequence_number ();
    }
    public int add_subfigure (int i)
    {
	DirectoryEntry  DE = append_directory_entry (308, 0);
	ParameterData PD = DE.data();
	
	DE.set_subordinate (StatusSubordinate.ss_PhysicallyDependent);
	DE.set_hierarchy (StatusHierarchy.sh_GlobalDefer);
	
	PD.append_ld (0);
	PD.append_HS (String.format("Region-%d",i));
	PD.append_ld (1);
	PD.append_ld (i);
	
	return DE.sequence_number();
    }
    public int add_subfigure_instance (int K)
    {
	DirectoryEntry  DE = append_directory_entry (408, 0);
	ParameterData PD = DE.data();
	
	DE.set_hierarchy (StatusHierarchy.sh_GlobalDefer);
	
	PD.append_ld (K);
	PD.append_ld (0);
	PD.append_ld (0);
	PD.append_ld (0);
	PD.append_ld (1);
	
	return DE.sequence_number();
    }

    public void add_manifold_solids (RegionLoader rl)
    {
	int Nj = rl.region_vector().size ();
	for (int j = 0; j < Nj; j++)
	    {
		Region r = rl.region_vector().elementAt (j);

		int K = add_subfigure (add_manifold_solid (r, rl));
		add_subfigure_instance (K);
	    }
    }
}
