/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

public enum PlaneCompare
{
	XP(Plane.X,true), XN(Plane.X,false),
	YP(Plane.Y,true), YN(Plane.Y,false),
	ZP(Plane.Z,true), ZN(Plane.Z,false);

	private Plane    m_plane;
	private boolean  m_parity;

	PlaneCompare (Plane plane, boolean parity)
	{
	    m_plane  = plane;
	    m_parity = parity;
	}

	public boolean is_internal (double offset, Vertex V)
	{
	    boolean bInternal = true;

	    switch (m_plane)
		{
		case X:
		    if (m_parity)
			{
			    if (V.xd () > offset)
				bInternal = false;
			}
		    else
			{
			    if (V.xd () < offset)
				bInternal = false;
			}
		    break;
		case Y:
		    if (m_parity)
			{
			    if (V.yd () > offset)
				bInternal = false;
			}
		    else
			{
			    if (V.yd () < offset)
				bInternal = false;
			}
		    break;
		case Z:
		    if (m_parity)
			{
			    if (V.zd () > offset)
				bInternal = false;
			}
		    else
			{
			    if (V.zd () < offset)
				bInternal = false;
			}
		    break;
		}
	    return bInternal;
	}
	public boolean is_external (double offset, Vertex V)
	{
	    boolean bExternal = true;

	    switch (m_plane)
		{
		case X:
		    if (m_parity)
			{
			    if (V.xd () < offset)
				bExternal = false;
			}
		    else
			{
			    if (V.xd () > offset)
				bExternal = false;
			}
		    break;
		case Y:
		    if (m_parity)
			{
			    if (V.yd () < offset)
				bExternal = false;
			}
		    else
			{
			    if (V.yd () > offset)
				bExternal = false;
			}
		    break;
		case Z:
		    if (m_parity)
			{
			    if (V.zd () < offset)
				bExternal = false;
			}
		    else
			{
			    if (V.zd () > offset)
				bExternal = false;
			}
		    break;
		}
	    return bExternal;
	}
	public boolean intersects (double offset, Edge E, int vid)
	{
	    Vertex V1 = E.v1 ();
	    Vertex V2 = E.v2 ();

	    boolean bIntersects = true;

	    E.set_internal (false);
	    E.set_external (false);

	    if (is_internal (offset, V1) && is_internal (offset, V2))
		{
		    E.set_internal (true);
		    bIntersects = false;
		}
	    if (is_external (offset, V1) && is_external (offset, V2))
		{
		    E.set_external (true);
		    bIntersects = false;
		}
	    if (!bIntersects)
		{
		    E.set_intersects (false);
		    E.set_intersection (null);
		    return false;
		}

	    Vertex Vi = new Vertex (vid);

	    double r, x, y, z;

	    switch (m_plane)
		{
		case X:
		    r = (offset - V1.xd ()) / (V2.xd () - V1.xd ());
		    y = V1.yd () + r * (V2.yd () - V1.yd ());
		    z = V1.zd () + r * (V2.zd () - V1.zd ());
		    Vi.set_coordinates (offset, y, z);
		    break;
		case Y:
		    r = (offset - V1.yd ()) / (V2.yd () - V1.yd ());
		    x = V1.xd () + r * (V2.xd () - V1.xd ());
		    z = V1.zd () + r * (V2.zd () - V1.zd ());
		    Vi.set_coordinates (x, offset, z);
		    break;
		case Z:
		    r = (offset - V1.zd ()) / (V2.zd () - V1.zd ());
		    y = V1.yd () + r * (V2.yd () - V1.yd ());
		    x = V1.xd () + r * (V2.xd () - V1.xd ());
		    Vi.set_coordinates (x, y, offset);
		    break;
		}
	    E.set_intersects (true);
	    E.set_intersection (Vi);

	    return true;
	}
	public double normal_x ()
	{
	    double component = 0;

	    switch (m_plane)
		{
		case X:
		    if (m_parity)
			component =  1.0;
		    else
			component = -1.0;
		    break;
		case Y:
		    break;
		case Z:
		    break;
		}
	    return component;
	}
	public double normal_y ()
	{
	    double component = 0;

	    switch (m_plane)
		{
		case X:
		    break;
		case Y:
		    if (m_parity)
			component =  1.0;
		    else
			component = -1.0;
		    break;
		case Z:
		    break;
		}
	    return component;
	}
	public double normal_z ()
	{
	    double component = 0;

	    switch (m_plane)
		{
		case X:
		    break;
		case Y:
		    break;
		case Z:
		    if (m_parity)
			component =  1.0;
		    else
			component = -1.0;
		    break;
		}
	    return component;
	}
	public PlaneCompare reverse ()
	{
	    PlaneCompare pc = null;
	    //. error pc might not be initialized 

	    switch (m_plane)
		{
		case X:
		    if (m_parity)
			pc = XN;
		    else
			pc = XP;
		    break;
		case Y:
		    if (m_parity)
			pc = YN;
		    else
			pc = YP;
		    break;
		case Z:
		    if (m_parity)
			pc = ZN;
		    else
			pc = ZP;
		    break;
		}
	    return pc;
	}
}