/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

import java.util.*;

public class Region{
    private String point; 
    private int vertices;
    private boolean internal;

    private Vector<Facet>   m_facets = null;
    private Vector<EdgeKey> m_edgekeys = null;

    private Region m_rext = null;
    private boolean m_bKeepExternal = false;

    private float m_r;
    private float m_g;
    private float m_b;
    private String m_color_name = null;

    private boolean m_bInternal;
    private boolean m_bExternal;
    private boolean m_bIntersects;

    private double m_facet_area_min;
    private double m_facet_area_max;
    private double m_volume;

    public Region(String p, int v, int f, boolean i)
    {
	point = p;
	vertices = v;
	internal = i;
        m_facets = new Vector<Facet> ();
	m_edgekeys = new Vector<EdgeKey> ();
    }
    public String get_point()
    {
	return point;
    }
    public int get_vertices_count()
    {
	return vertices;
    }
    public int get_facets_count()
    {
	return m_facets.size ();
    }
    public boolean get_region_binternal()
    {
	return internal;
    }
    
    private void add_facet(Facet f)
    {
	m_facets.addElement (f);
    }
    public void add_facet(Vector r, Vector v)
    {
	add_facet (new Facet (r, v));
    }
    public void map_facets (TreeMap<String,FacetPair> facet_map)
    {
	int length = m_facets.size ();

	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt (i);

		String key = f.make_ID ();

		if (facet_map.containsKey (key))
		    {
			FacetPair fp = facet_map.get (key);
			fp.add_pair (f, m_r, m_g, m_b);
		    }
		else
		    {
			facet_map.put (key, new FacetPair (f, m_r, m_g, m_b));
		    }
	    }
    }
    public void map_edges (TreeMap<EdgeKey,Edge> edge)
    {
	int length = m_facets.size ();

	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt (i);
		f.map_edges (edge);
	    }
	if (m_bKeepExternal)
	    m_rext.map_edges (edge);
    }
    public void make_facet_edges (TreeMap<EdgeKey,Edge> edge)
    {
	int length = m_facets.size ();

	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt (i);
		f.make_facet_edges (edge, true);
	    }
	if (m_bKeepExternal)
	    m_rext.make_facet_edges (edge);
    }
    public final Vector<Facet> region_facet_v ()
    {
	return m_facets;
    }
    public float color_r ()
    {
	return m_r;
    }
    public float color_g ()
    {
	return m_g;
    }
    public float color_b ()
    {
	return m_b;
    }
    public String color_name ()
    {
	return m_color_name;
    }
    public void set_color ( float r, float g, float b, String color_n)
    {
	m_r = r;
	m_g = g;
	m_b = b;
	m_color_name = color_n;
    }

    public void set_internal (boolean bInternal)
    {
	m_bInternal = bInternal;
    } 
    public boolean get_internal ()
    {
	return m_bInternal;
    }
    
    public void set_external (boolean bExternal)
    {
	m_bExternal = bExternal;
    }
    public boolean get_external ()
    {
	return m_bExternal;
    }
    
    public void set_intersects (boolean bIntersects)
    {
	m_bIntersects = bIntersects;
    }
    public boolean get_intersects ()
    {
	return m_bIntersects; 
    }
    public void check_facets ()
    {
	set_intersects(false);
	set_internal(true);
	set_external(true);
	
	int length = m_facets.size ();
	
	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt(i);
		f.check_edges();
		
		if(f.get_intersects())
		    {
			set_intersects(true);
			set_internal(false);
			set_external(false);
			continue;
		    }
		if (!f.get_external())
		    {
			set_external(false);
		    }
		if (!f.get_internal())
		    {
			set_internal(false);
		    }
	    }
	if (!get_internal() && !get_external())
	    {
		set_intersects(true);
	    }
    }
    public void add_missing_facet (PlaneCompare pc, TreeMap<EdgeKey,Edge> edge)
    {
	Facet f = new Facet (m_edgekeys, pc, edge);
	f.make_facet_edges (edge, false);
	add_facet (f);

	if (m_bKeepExternal)
	    {
		f = new Facet (m_edgekeys, pc.reverse (), edge);
		f.make_facet_edges (edge, false);
		m_rext.add_facet (f);
	    }
    }
    public void crop_to_plane (double offset, PlaneCompare pc, boolean bKeepExternal)
    {
	if (bKeepExternal)
	    {
		m_rext = new Region (point, 0, 0, true);
		m_rext.set_color (m_r, m_g, m_b, m_color_name);
	    }
	else
	    m_rext = null;

	m_bKeepExternal = bKeepExternal;

	m_edgekeys.clear ();

	int length = m_facets.size ();

	for (int i = length - 1; i >= 0; i--)
	    {
		Facet f = m_facets.elementAt(i);

		if (f.get_external())
		    {
			Facet f1 = m_facets.remove(i);
			if (m_bKeepExternal)
			    {
				m_rext.add_facet (f1);
			    }
			continue;
		    }
		if (f.get_internal())
		    {
			EdgeKey key = f.edge_on_plane (offset, pc); // method may return null
			if (key != null)
			    m_edgekeys.addElement(key);
			continue;
		    }
		if (f.get_intersects())
		    {
			EdgeKey key = f.crop_to_plane (offset, pc, m_bKeepExternal);
			m_edgekeys.addElement(key);
			if (m_bKeepExternal)
			    {
				Facet fext = f.get_external_facet ();
				m_rext.add_facet (fext);
			    }
		    }
	    }
    }
    public Region get_external_region () // only use once! (removes internal region)
    {
	m_bKeepExternal = false;
	return m_rext;
    }

    public void save (RegionLoader rl)
    {
	rl.write(String.format("  <Region point=\"%s\" vertices=\"%d\" facets=\"%d\" internal=\"%s\">\n",
			       point, vertices, m_facets.size(), internal));

	for (int j = 0; j < m_facets.size(); j++)
	    {
		Facet f = m_facets.elementAt(j);
		f.save(rl);
	    }
	rl.write("  </Region>\n");
    }
    public void check_edge_count ()
    {
	int length = m_facets.size ();

	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt (i);
		f.zero_edge_counts ();
	    }
	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt (i);
		f.mark_edges ();
	    }
	for (int i = 0; i < length; i++)
	    {
		Facet f = m_facets.elementAt (i);
		f.check_edge_count ();
	    }
    }

    public void calculate_stats ()
    {
	boolean bFirst = true;

	double x = 0;
	double y = 0;
	double z = 0;

	int count = m_facets.size ();

	for (int j = 0; j < count; j++)
	    {
		Facet f = m_facets.elementAt (j);
		f.calculate_center_and_area ();

		Vertex fc = f.center ();
		x += fc.xd ();
		y += fc.yd ();
		z += fc.zd ();

		double area = f.area ();
		if (bFirst)
		    {
			m_facet_area_min = area;
			m_facet_area_max = area;
			bFirst = false;
		    }
		else
		    {
			if (m_facet_area_min > area)
			    m_facet_area_min = area;
			if (m_facet_area_max < area)
			    m_facet_area_max = area;
		    }
	    }

	Vertex center = new Vertex (x / count, y / count, z / count);

	m_volume = 0;

	for (int j = 0; j < count; j++)
	    {
		Facet f = m_facets.elementAt (j);

		final Vertex fc = f.center ();

		double dx = fc.xd () - center.xd ();
		double dy = fc.yd () - center.yd ();
		double dz = fc.zd () - center.zd ();

		double height = Math.abs (dx * f.normal_xd () + dy * f.normal_yd () + dz * f.normal_zd ());

		double area = f.area ();

		m_volume += area * height / 3;
	    }
    }
    public void print_stats ()
    {
	System.out.print (String.format ("%E\t%d\t%E\t%E\n", m_volume, m_facets.size (), m_facet_area_min, m_facet_area_max));
    }
    public double volume ()
    {
	return m_volume;
    }
}
