/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

import javax.media.opengl.GL;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLCanvas;
import javax.media.opengl.GLEventListener;

import javax.media.opengl.glu.GLU;

import com.sun.opengl.util.Animator;

import java.util.*;

public class RegionCanvas implements GLEventListener
{
    private static final GLU m_glu = new GLU ();

    private RegionLoader m_loader = null;

    private GLCanvas m_canvas = null;
    private Animator m_animator = null;

    private float rotateX = 0.0f;
    private float rotateY = 0.0f;
    private float rotateZ = 0.0f;
    
    private float m_fovy   = 60.0f;
    private float m_aspect = 1.0f;
    private float m_zNear  = 1.0f;
    private float m_zFar   = 4.0f;

    private boolean m_bTransparent = false;
    
    public RegionCanvas (RegionLoader loader)
    {
	m_loader =  loader;

	m_canvas = new GLCanvas ();
	m_canvas.addGLEventListener (this);
    }

    public GLCanvas canvas ()
    {
	return m_canvas;
    }

    public void set_transparent (boolean bTransparent)
    {
	m_bTransparent = bTransparent;
    }

    public void start_animation ()
    {
	if (m_animator == null)
	    {
		// the Animator is used to constantly update the screen
		m_animator = new Animator (m_canvas);
	    }
	m_animator.start ();
    }
    public void stop_animation ()
    {
	if (m_animator != null)
	    {
		m_animator.stop();
	    }
    }

    public void init (GLAutoDrawable drawable)
    {
	// the GL class of JOGL - retrieve from GLAutoDrawable and pass through stack for thread safety
	final GL gl = drawable.getGL ();
	
	// The init() method is called when a new OpenGL context is created for the given GLAutoDrawable.
	// Note: The init() method may be called more than once during the lifetime of the application.
	
	gl.glClearColor (0.0f, 0.0f, 0.0f, 0.0f); // r,g,b,a; sets default background color
//	gl.glClearColor (1.0f, 1.0f, 1.0f, 0.0f); // r,g,b,a; sets default background color
        gl.glClearDepth (1.0f); // clear value for the depth buffer
	
        gl.glEnable (GL.GL_DEPTH_TEST);
        gl.glDepthFunc (GL.GL_LEQUAL);  // value used for depth buffer comparisons - see glEnable (GL_DEPTH_TEST)
	
	gl.glShadeModel (GL.GL_SMOOTH); // symbolic value representing a shading technique: GL_FLAT and GL_SMOOTH.
	
        gl.glHint (GL.GL_PERSPECTIVE_CORRECTION_HINT, GL.GL_NICEST); // target, mode

	// Lighting model:

        gl.glEnable (GL.GL_LIGHTING);
        gl.glEnable (GL.GL_LIGHT0);
	
	gl.glColorMaterial (GL.GL_FRONT_AND_BACK, GL.GL_AMBIENT_AND_DIFFUSE);
        gl.glEnable (GL.GL_COLOR_MATERIAL);

	if (m_bTransparent)
	    {
		gl.glEnable (GL.GL_BLEND);
		gl.glBlendFunc (GL.GL_SRC_ALPHA, GL.GL_ONE_MINUS_SRC_ALPHA);
	    }
    }

    public void display (GLAutoDrawable drawable)
    {
	// the GL class of JOGL - retrieve from GLAutoDrawable and pass through stack for thread safety
	final GL gl = drawable.getGL ();
	
	// The display() method is called to perform per-frame rendering.

	gl.glClear (GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT); // clear buffers to preset values

	// gl.glLoadIdentity (); // glLoadIdentity replaces the current matrix with the identity matrix.

        gl.glMatrixMode (GL.GL_PROJECTION);
        gl.glLoadIdentity ();
	
        m_glu.gluPerspective (m_fovy, m_aspect, m_zNear, m_zFar);
	
	gl.glTranslatef (0.0f, 0.0f, -2.5f); // x, y, z
	
        gl.glRotatef (rotateX, 1.0f, 0.0f, 0.0f); // angle, x, y, z; rotation of angle degrees around the vector (x)
        gl.glRotatef (rotateY, 0.0f, 1.0f, 0.0f);
        gl.glRotatef (rotateZ, 0.0f, 0.0f, 1.0f);

        rotateX += 0.16f;
        rotateY += 0.04f;
        rotateZ += 0.01f;

	Collection<FacetPair> fpc = m_loader.facet_pairs ();
	Iterator<FacetPair> fpi = fpc.iterator ();
	while (fpi.hasNext ())
	    {
		FacetPair fp = fpi.next ();

		if (!m_bTransparent && fp.is_paired ()) continue; // no point drawing internal faces

		gl.glNormal3f (fp.normal_xf (), fp.normal_yf (), fp.normal_zf ());

		Vector vertices = fp.vertices ();

		int vertex_count = vertices.size ();

		gl.glBegin (GL.GL_POLYGON); // fill facet

		if (m_bTransparent)
		    gl.glColor4f (fp.color_r (), fp.color_g (), fp.color_b (), 0.7f);
		else
		    gl.glColor3f (fp.color_r (), fp.color_g (), fp.color_b ());

		for (int vc = 0; vc < vertex_count; vc++)
		    {
			Vertex v = (Vertex) vertices.elementAt (vc);

			gl.glVertex3f (v.xf (), v.yf (), v.zf ());
		    }
		gl.glEnd ();
	    }

	return;
	/* old region-by-region method:
	
	Vector regions = m_loader.region_vector ();

	int region_count = regions.size ();
	
	for (int rc = 0; rc < region_count; rc++)
	    {
		Region r = (Region) regions.elementAt (rc);

		Vector facets = r.region_facet_v ();

		int facet_count = facets.size ();

		for (int fc = 0; fc < facet_count; fc++)
		    {
			Facet f = (Facet) facets.elementAt (fc);

			gl.glNormal3f (f.normal_xf (), f.normal_yf (), f.normal_zf ());

			Vector vertices = f.vertices ();

			int vertex_count = vertices.size ();

			gl.glBegin (GL.GL_POLYGON); // fill facet

			if (m_bTransparent)
			    gl.glColor4f (r.color_r (), r.color_g (), r.color_b (), 0.7f);
			else
			    gl.glColor3f (r.color_r (), r.color_g (), r.color_b ());

			for (int vc = 0; vc < vertex_count; vc++)
			    {
				Vertex v = (Vertex) vertices.elementAt (vc);

				gl.glVertex3f (v.xf (), v.yf (), v.zf ());
			    }
			gl.glEnd ();

			gl.glBegin (GL.GL_LINE_STRIP); // Draw edges

			gl.glColor3f (0.0f, 0.0f, 0.0f);

			Vertex v0 = null;

			for (int vc = 0; vc < vertex_count; vc++)
			    {
				Vertex v = (Vertex) vertices.elementAt (vc);

				if (vc == 0) v0 = v;

				gl.glVertex3f (v.xf (), v.yf (), v.zf ());
			    }
			gl.glVertex3f (v0.xf (), v0.yf (), v0.zf ());
			gl.glEnd ();
		    }
	    } */
    }

    public void displayChanged (GLAutoDrawable drawable, boolean modeChanged, boolean deviceChanged)
    {
	// the GL class of JOGL - retrieve from GLAutoDrawable and pass through stack for thread safety
	// final GL gl = drawable.getGL ();

	// The displayChanged() method is designed to allow applications to support on-the-fly screen
	// mode switching, but support for this is not yet implemented so the body of this method should
	// remain empty.
    }

    public void reshape (GLAutoDrawable drawable, int x, int y, int width, int height)
    {
	// the GL class of JOGL - retrieve from GLAutoDrawable and pass through stack for thread safety
	// final GL gl = drawable.getGL ();

	// update aspect ratio
	if (height > 0)
	    m_aspect = (float) width / (float) height;
	else
	    m_aspect = (float) width;

	// The reshape() method is called when the drawable has been resized; the default implementation
	// automatically resizes the OpenGL viewport so often it is not necessary to do any work in this
	// method. 

	// if (height <= 0) height = 1;

        // final float h = (float) width / (float) height;

	// The matrix generated by gluPerspective() is multipled by the current matrix, just as if
	// glMultMatrix() were called with the generated matrix. To load the perspective matrix onto the
	// current matrix stack instead, precede the call to gluPerspective() with a call to
	// glLoadIdentity().

	// glMatrixMode() specifies which matrix stack is the target for subsequent matrix operations.
	// Three values are accepted: GL_MODELVIEW, GL_PROJECTION, and GL_TEXTURE. 

        // gl.glMatrixMode (GL.GL_PROJECTION);
        // gl.glLoadIdentity ();

        // m_glu.gluPerspective (50.0f, h, 1.0, 1000.0); // fovy, aspect, zNear, zFar

	// fovy    Specifies the field of view angle, in degrees, in the y direction.
	// aspect  Specifies the aspect ratio that determines the field of view in the x direction.
	//         The aspect ratio is the ratio of x (width) to y (height).
	// zNear   Specifies the distance from the viewer to the near clipping plane (always positive).
	// zFar    Specifies the distance from the viewer to the far  clipping plane (always positive).
 
        // gl.glMatrixMode (GL.GL_MODELVIEW);
        // gl.glLoadIdentity ();
    }
}
