/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

import java.net.*;
import java.io.*;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import java.util.*;
import java.util.TreeMap.*;

public class RegionLoader
{
    private int total_points;
    private int total_vertices;
    private int total_regions;

    private int m_vid_internal = -1; // this actually reserves -1 itself for improper vertices

    private Vector<Vertex> vertex;
    private Vector<Region> region;

    private Vector<Region> new_region;// adding new vector for intersecting regions

    private Region last;

    private TreeMap<String,Point>     m_point = null;
    private TreeMap<String,FacetPair> m_facet = null;
    private TreeMap<EdgeKey,Edge>     m_edge  = null;

    private Collection<FacetPair> m_facet_pairs = null;
    private Collection<Edge>      m_edge_unique = null;

    private int iges_vertex_list_de_seqno;
    private int iges_edge_list_de_seqno;
    
    private int m_rr = 0;
    private int m_rg = 0;
    private int m_rb = 0;

    private FileOutputStream m_fout = null;

    RegionLoader ()
    {
	vertex = new Vector<Vertex> ();
	region = new Vector<Region> ();
	new_region = new Vector<Region> ();

	m_point = new TreeMap<String,Point> ();
	m_facet = new TreeMap<String,FacetPair> ();
	m_edge  = new TreeMap<EdgeKey,Edge> ();
    }

    public int next_vid ()
    {
	return --m_vid_internal;
    }    
    public void points (int count)
    {
	total_points = count;
    }
    public void vertices (int count)
    {
	total_vertices = count;
    }

    public void add_point (String vid, double x, double y, double z)
    {
	Point v = new Point (vid,x,y,z);
	m_point.put (vid, v);
    }
    public void add_vertex (int vid, double x, double y, double z, boolean bInternal)
    {
	Vertex v = new Vertex(vid,x,y,z,bInternal);
	vertex.addElement(v);
    }
    
    public void regions(int rcount)
    {
	total_regions = rcount;
    }
    public void add_region (String point, int v, int i, boolean internal)
    {
	Region r = new Region (point,v,i,internal);
	region.addElement (r);
	last = r;

	r.set_color (m_rr / 5.0f, m_rg / 5.0f, m_rb / 5.0f, String.format ("RGB%d%d%d", m_rr, m_rg, m_rb));

	m_rr++;
	if (m_rr > 5)
	    {
		m_rr = 0;
		m_rg++;
		if (m_rg > 5)
		    {
			m_rg = 0;
			m_rb++;
			if (m_rb > 5)
			    m_rb = 0;
		    }
	    }
    }

    public void add_facet(Vector f)
    {
	last.add_facet (f, vertex);
    }

    public void map_facets ()
    {
	m_facet.clear ();

	int length = region.size ();

	for (int i = 0; i < length; i++)
	    {
		Region r = region.elementAt (i);
		r.map_facets (m_facet);
	    }
	m_facet_pairs = m_facet.values ();
    }

    public void map_edges ()
    {
	m_edge.clear ();

	int length = region.size ();

	for (int i = 0; i < length; i++)
	    {
		Region r = region.elementAt (i);
		r.map_edges (m_edge);
	    }
	m_edge_unique = m_edge.values ();
    }
    
    public void make_facet_edges ()
    {
	int length = region.size ();

	for (int i = 0; i < length; i++)
	    {
		Region r = region.elementAt (i);
		r.make_facet_edges (m_edge);
	    }
    }
    
    public final TreeMap<String,Point> point_map ()
    {
	return m_point;
    }
    public final TreeMap<String,FacetPair> facet_map ()
    {
	return m_facet;
    }
    public final TreeMap<EdgeKey,Edge> edge_map ()
    {
	return m_edge;
    }

    public final Collection<FacetPair> facet_pairs ()
    {
	return m_facet_pairs;
    }
    public final Collection<Edge> edges ()
    {
	return m_edge_unique;
    }

    public final Vector<Region> region_vector ()
    {
	return region;
    }
    public final Vector<Vertex> vertex_vector()
    {
	return vertex;
    }
    public void set_iges_vertex_list_de_seqno (int i)
    {
	iges_vertex_list_de_seqno = i;
    }
    public int get_iges_vertex_list_de_seqno ()
    {
	return iges_vertex_list_de_seqno;
    }
    public void set_iges_edge_list_de_seqno(int i)
    {
	iges_edge_list_de_seqno = i;
    }
    public int get_iges_edge_list_de_seqno()
    {
	return iges_edge_list_de_seqno;
    }
    
    public void crop_to_plane (double offset, PlaneCompare pc, boolean bKeepExternal)
    {
	Collection<Edge> collection = m_edge.values ();
	Iterator<Edge> i = collection.iterator ();

	int vid = next_vid ();
	
	while (i.hasNext ())
	    {
		Edge e = i.next ();

		if (pc.intersects (offset, e, vid))
		    {
			vertex.addElement (e.get_intersection ());
			vid = next_vid ();
		    }
	    }

	// 1. mark all regions and facets as internal/external/intersecting
	// 2. remove all external regions
	// 3. remove external facets from intersecting regions
	// 4. the tricky bit...
	int length = region.size();

	for (int j = length - 1; j >= 0; j--)
	    {
		Region r = region.elementAt(j);
		r.check_facets();
		if (r.get_external())
		    {
			if (!bKeepExternal)
			    region.remove (j); //region are not removed we are retaining them.
			continue;
		    }
		if (r.get_intersects())
		    {
			// this will destroy a lot of edge information
			r.crop_to_plane (offset, pc, bKeepExternal);
		    }
	    }

	// recreate edge maps and directed edges
	map_edges ();
	make_facet_edges ();

	length = region.size();
	
	// System.out.print ("... Edge Checks ...\n");
	for (int j = 0; j < length ; j++)
	    {
		Region r = region.elementAt(j);
		
		if (r.get_intersects())
		    {
			r.add_missing_facet (pc, m_edge);
			if (bKeepExternal)
			    {
				Region rext = r.get_external_region ();
				region.addElement (rext);
			    }
		    }
		
		// System.out.print (String.format ("Region %d, checking edges...\n", j));
		// r.check_edge_count ();
	    }

	map_facets ();
    }
    
    public void crop_to_cube (double offset)
    {
	crop_to_plane ( offset, PlaneCompare.XP, false);
	crop_to_plane (-offset, PlaneCompare.XN, false);
        crop_to_plane ( offset, PlaneCompare.YP, false);
	crop_to_plane (-offset, PlaneCompare.YN, false);
	crop_to_plane ( offset, PlaneCompare.ZP, false);
	crop_to_plane (-offset, PlaneCompare.ZN, false);
    }

    public void write (String text)
    {
	if (m_fout == null) return;

	try {
	    m_fout.write (text.getBytes ());
	} catch (Exception e) {
	    System.err.println ("RegionLoader::save() Exception caught while writing file!\n");
	    System.err.println (e);
	}
    }

    public void save (String filename)
    {
	File f = new File (filename);
	try {
	    m_fout = new FileOutputStream (f);
	} catch (Exception e) {
	    System.err.println ("RegionLoader::save() Exception caught while opening file!\n");
	    System.err.println (e);
	}

	write ("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n");
	write ("<Voronoi-set>\n");
	write (String.format("<Points count =\"%d\">\n",total_points));
	Collection<Point> collection = m_point.values ();
	Iterator<Point> i = collection.iterator ();
	
	while (i.hasNext ())
	    {
		Point p = i.next ();
		write (String.format ("  <point id=\"%s\" x=\"" +
				      Double.toString (p.x()) + "\" y=\"" +
				      Double.toString (p.y()) + "\" z=\"" +
				      Double.toString (p.z()) + "\"/>\n", p.pid()));
	    }
	write(" </Points>\n");
	write(String.format(" <Vertices count =\"%d\">\n",total_vertices));

	for (int j=0; j<vertex.size(); j++)
	    {   
		Vertex V = vertex.elementAt(j);
		write (String.format ("  <Vertex id=\"%d\" x=\"" +
				      Double.toString (V.xd()) + "\" y=\"" +
				      Double.toString (V.yd()) + "\" z=\"" +
				      Double.toString (V.zd()) + "\" internal=\"%s\"/>\n", V.vid(), V.bInternal()));
	    }
	write(" </Vertices>\n");
		
	write(String.format(" <Regions count=\"%d\">\n",region.size()));
	
	for ( int j=0; j<region.size(); j++)
	    {
		Region r = region.elementAt(j);
		r.save (this);
	    }
	write(" </Regions>\n");
	// regions

	write ("</Voronoi-set>\n");

	try {
	    m_fout.close ();
	} catch (Exception e) {
	    System.err.println ("RegionLoader::save() Exception caught while closing file!\n");
	    System.err.println (e);
	}
	m_fout = null;
    }

    public void stats ()
    {
	int count = region.size ();

	double volume_mean = 0;
	double volume_min = 0;
	double volume_max = 0;
	double volume_rms = 0;
	double volume = 0;

	boolean bFirst = true;

	for (int j = 0; j < count; j++)
	    {
		Region r = region.elementAt (j);
		r.calculate_stats ();
		volume = r.volume ();
		volume_mean += volume;
		if (bFirst)
		    {
			volume_min = volume;
			volume_max = volume;
			bFirst = false;
		    }
		else
		    {
			if (volume_min > volume)
			    volume_min = volume;
			if (volume_max < volume)
			    volume_max = volume;
		    }
	    }
	volume_mean = volume_mean / count;

	for (int j = 0; j < count; j++)
	    {
		Region r = region.elementAt (j);
		volume = r.volume () - volume_mean;
		volume_rms += volume * volume;
	    }
	volume_rms = Math.sqrt (volume_rms / count);

	System.out.print (String.format ("Region volume statistics:\t\tcount:\t%d\n", count));
	System.out.print (String.format ("mean\tmin\tmax\tr.m.s.\n"));
	System.out.print (String.format ("%E\t%E\t%E\t%E\n", volume_mean, volume_min, volume_max, volume_rms));

	double area_mean = 0;
	double area_min  = 0;
	double area_max  = 0;
	double area_rms  = 0;
	double area      = 0;

	bFirst = true;

	int fp_count = m_facet_pairs.size ();

	Iterator<FacetPair> fpi = m_facet_pairs.iterator ();

	while (fpi.hasNext ())
	    {
		FacetPair fp = fpi.next ();

		area = fp.area ();
		area_mean += area;
		if (bFirst)
		    {
			area_min = area;
			area_max = area;
			bFirst = false;
		    }
		else
		    {
			if (area_min > area)
			    area_min = area;
			if (area_max < area)
			    area_max = area;
		    }
	    }
	area_mean = area_mean / fp_count;

	fpi = m_facet_pairs.iterator ();

	while (fpi.hasNext ())
	    {
		FacetPair fp = fpi.next ();
		area = fp.area () - area_mean;
		area_rms += area * area;
	    }
	area_rms = Math.sqrt (area_rms / fp_count);

	System.out.print (String.format ("Facet area statistics:\t\tcount:\t%d\n", fp_count));
	System.out.print (String.format ("mean\tmin\tmax\tr.m.s.\n"));
	System.out.print (String.format ("%E\t%E\t%E\t%E\n", area_mean, area_min, area_max, area_rms));

	System.out.print (String.format ("Region volume; facet min & max areas:\t\t\t\n"));
	System.out.print (String.format ("volume\t#facets\tmin\tmax\n"));

	for (int j = 0; j < count; j++)
	    {
		Region r = region.elementAt (j);
		r.print_stats ();
	    }

	// stats of an intersection

	crop_to_plane (0, PlaneCompare.ZP, false);

	int fp_count_in_plane = 0;

	area_mean = 0;
	area_min  = 0;
	area_max  = 0;
	area_rms  = 0;
	area      = 0;

	bFirst = true;

	fpi = m_facet_pairs.iterator ();

	while (fpi.hasNext ())
	    {
		FacetPair fp = fpi.next ();

		if (!fp.in_plane (0, PlaneCompare.ZP)) continue;
		++fp_count_in_plane;

		fp.f1().calculate_center_and_area ();

		area = fp.area ();
		area_mean += area;
		if (bFirst)
		    {
			area_min = area;
			area_max = area;
			bFirst = false;
		    }
		else
		    {
			if (area_min > area)
			    area_min = area;
			if (area_max < area)
			    area_max = area;
		    }
	    }
	area_mean = area_mean / fp_count_in_plane;

	fpi = m_facet_pairs.iterator ();

	while (fpi.hasNext ())
	    {
		FacetPair fp = fpi.next ();

		if (!fp.in_plane (0, PlaneCompare.ZP)) continue;

		area = fp.area () - area_mean;
		area_rms += area * area;
	    }
	area_rms = Math.sqrt (area_rms / fp_count_in_plane);

	System.out.print (String.format ("Intersection: Facet area statistics:\t\tcount:\t%d\n", fp_count_in_plane));
	System.out.print (String.format ("mean\tmin\tmax\tr.m.s.\n"));
	System.out.print (String.format ("%E\t%E\t%E\t%E\n", area_mean, area_min, area_max, area_rms));

	System.out.print (String.format ("Intersection: Facet area & vertex/edge count:\t\t\t\n"));
	System.out.print (String.format ("area\t#edges\t\t\n"));

	fpi = m_facet_pairs.iterator ();

	while (fpi.hasNext ())
	    {
		FacetPair fp = fpi.next ();

		if (!fp.in_plane (0, PlaneCompare.ZP)) continue;

		area = fp.area ();

		int vertex_count = fp.vertices().size ();

		System.out.print (String.format ("%E\t%d\t\t\n", area, vertex_count));
	    }
    }

    public static RegionLoader load (String filename)
    {
	RegionLoader loader = new RegionLoader ();
	
	try {
	    File file = new File (filename);
	    
	    SAXParserFactory factory = SAXParserFactory.newInstance ();
	    try {
		SAXParser saxParser = factory.newSAXParser ();
		saxParser.parse (file, new RegionHandler (loader));

	    } catch (Throwable err) {
		// err.printStackTrace ();
		loader = null;
	    }
	} catch (Exception e) {
	    loader = null;
	}
	if (loader == null) return null;

	loader.map_edges ();
	loader.make_facet_edges ();
	loader.map_facets ();

//	loader.crop_to_plane ( 0.01, PlaneCompare.ZP, false);
//	loader.crop_to_plane (-0.01, PlaneCompare.ZN, false);

	return loader;
    }

    public static RegionLoader load_uri (String uri)
    {
	RegionLoader loader = new RegionLoader ();
	
	try {
	    SAXParserFactory factory = SAXParserFactory.newInstance ();
	    try {
		SAXParser saxParser = factory.newSAXParser ();
		saxParser.parse (uri, new RegionHandler (loader));

	    } catch (Throwable err) {
		err.printStackTrace ();
		loader = null;
	    }
	} catch (Exception e) {
	    loader = null;
	}
	if (loader == null) return null;

	loader.map_edges ();
	loader.make_facet_edges ();
	loader.map_facets ();

	return loader;
    }
}
