/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* Copyright 2008 The University of Newcastle upon Tyne
 * 
 * This file is part of RegionReader.
 * 
 * RegionReader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * RegionReader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with RegionReader.  If not, see <http://www.gnu.org/licenses/>.
 */

/* program to calculate random points and finding voronoi vertices for them and writing in xml , rbox file */

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <math.h>

#include <vector>
#include <iostream>

class Coordinate
{
public:
	double x;
	double y;
	double z;

	bool bInternal;	
	bool bVertex;
		
    static double random_0_1 (); /* random value in range (0,1) */

    void random_coordinate ();

	double shortest_distance (Coordinate & C) const;

	bool check_internal ()
	{
		bInternal = false;

		if (x > -1.5)
			if (x < 1.5)
				if (y > -1.5)
					if (y < 1.5)
						if (z > -1.5)
							if (z < 1.5)
								bInternal = true;

		return bInternal;
	}

	Coordinate () :
    	x(0),
		y(0),
		z(0),
		bInternal(true),
		bVertex(false)
    {
    	// ...
    }
	Coordinate (double r, double s, double t) :
		x(r),
		y(s),
		z(t)
	{
		check_internal ();
	}

    ~Coordinate ()
    {
    }
};

double Coordinate::random_0_1 ()
{
	return (  ((double)rand()+(double)(1)) / ((double)(RAND_MAX)+(double)(2)) );
}

void Coordinate::random_coordinate () /*function declared for producing random numbers*/
{
	x = random_0_1 () - 0.5;
	y = random_0_1 () - 0.5;
	z = random_0_1 () - 0.5;
}

double Coordinate::shortest_distance (Coordinate & C) const /* function for shortest distance */
{
	bool bFirst = true;

	double distance = 0;

	for (int j=-1;j<=1;j++)
		for (int  k=-1;k<=1;k++)
			for (int l=-1;l<=1;l++)
			{
				double dx = C.x + ((double) l) - x;
				double dy = C.y + ((double) k) - y;
				double dz = C.z + ((double) j) - z;

				double dd = dx * dx + dy * dy + dz * dz;
				
				if (bFirst)
				{
					distance = dd;
					bFirst = false;
				}
				else if (distance > dd)
				{
					distance = dd;
				}
	 		}
	return sqrt (distance);
}

class Facet 
{
public:
	std :: vector<int> vertex;
	
	Facet ()
	{
		// ...
	}

	~Facet ()
	{
		// ...
	}
};

class Region 
{
public:
	std :: vector<int> vertex;

	int point; // index of point this region belongs to

	bool bRegionIsInternal;
	
	bool check_internal (Coordinate * const & V);
	
	double distance_from_centre (const Coordinate & C) const; // TODO
	
	Coordinate Centre;

	Facet * F;

	int facets;

   	Region () :
   		point(0),
   		bRegionIsInternal(true),
   		F(0),
		facets(0)
	{
		//...
	}

	~Region ()
	{
		if (F) delete [] F;
	}

	void create_facets (int count)
	{
		if (F) delete [] F;
		F = new Facet[count];
		facets = count;
	}
};

bool Region::check_internal (Coordinate * const & V)
{
	bRegionIsInternal = true;

	for (int l = 0; l < vertex.size (); l++)
		{ 
			if (!V[vertex[l]].bInternal)
				{
					bRegionIsInternal = false;
					break;
				}
		}
	return bRegionIsInternal;
}

class Generator
{
public:
	Coordinate * C;
	Coordinate * V;
	
	Region * R;
	
	int points;   // in C
	int vertices; // in V
	int regions;  // in R
	
	int generate_shortest (char * );
	int read_vertices ();
	int generate_xml(char *);
	int read_regions (char *);

	Generator (int number_of_points) :
		C(new Coordinate[number_of_points]),
		V(0),
		R(0),
		points(number_of_points)
	{
		// ...
	}

	~Generator ()
	{
		if (C) delete [] C;
		if (V) delete [] V;
		if (R) delete [] R;
	}
};

double Region :: distance_from_centre ( const Coordinate & C ) const // function to find distance between centre and point
{
	
				double dx = C.x  - Centre.x;
				double dy = C.y  - Centre.y;
				double dz = C.z  - Centre.z;

				double dd = dx * dx + dy * dy + dz * dz;
				
						
	return dd;
}
 

int main (int argc, char ** argv)   /* THIS IS MAIN FUNCTION*/
{
	if(argc != 5)
        {
			std::cout<<"usage: " << argv[0] << " <number of points> <random seed> <xml filename> <rbox filename>\n";		
			return -1;
		}

	int points = 0;
	sscanf(argv[1], "%d", &points);	         	

	if (points <= 0)
		{
			std::cout<<"usage:points should be > 0\n<number of points> <random seeds >\n";
			return -1;
		}
	
	long seed = 0;
	sscanf(argv[2], "%ld", &seed);/* choose a seed value */
	if ((seed <= 1) || (seed > RAND_MAX))
		{
			std::cout << "error: 1 < seed <= " << RAND_MAX << std::endl;
			return -1;
		}
	srand (seed);

	char * filename_xml  = argv[3];
	char * filename_rbox = argv[4];

	Generator G(points);

	if (G.generate_shortest (filename_rbox) != 0)
		goto _End;

    char buf[256 + strlen (filename_rbox)];
	sprintf (buf, "cat %s | qvoronoi o TO results.txt", filename_rbox);
	system (buf);

	if (G.read_vertices () != 0)
		goto _End;
		
			
	if (G.read_regions (filename_rbox) != 0)
		goto _End;
		
	if (G.generate_xml (filename_xml)!=0)
		goto _End;

_End:
	return 0;
}

int Generator::generate_shortest (char * filename_rbox)// GENERATING XML AND RBOX FILE USING RANDOM FUNCTIONS
{
		
	FILE * out_rbox = fopen (filename_rbox, "w"); // open rbox file for writing
		
	if (!out_rbox)
		{
			fprintf (stderr, "Failed to open rbox file \"%s\" for writing!\n", filename_rbox);
			return -1;
		}

	double minimum_distance = 1 / (double) (2 * points);

	int count = 0;

    while (count < points)
		{
			C[count].random_coordinate ();

			bool bPointOkay = true;

			for (int i = 0; i < count; i++)
				if ( C[count].shortest_distance ( C[i] ) < minimum_distance)
					{
						bPointOkay = false;
						break;
					}
			     
				
			    ++count;
		}

	fprintf (out_rbox, "3 rbox %d\n%d\n",points * 27,points * 27);
	
	int s = 0;

	for (int j=-1;j<=1;j++)
		for (int  k=-1;k<=1;k++)
			for (int l=-1;l<=1;l++)
				for (int i = 0; i < points; i++)
					{
						Coordinate point;

						point.x = C[i].x + l;
						point.y = C[i].y + k;
						point.z = C[i].z + j;
								   
						fprintf (out_rbox, " %.16lf\t %.16lf\t %.16lf\n", point.x, point.y, point.z);
						
						s++;
					}

	fclose (out_rbox); // closing rbox output file

	return 0;
}

int Generator::read_vertices () // Function Read_Vertices begins
{
	char * filename_input = "results.txt";

	FILE * in = fopen (filename_input, "r"); // opening a file for reading
   
    if (!in) // checking it has opened okay
		{
			fputs ("Failed to open file for reading!\n", stderr);
			return -1;
		}

	int dimensions, n;
    bool bFileReadOkay = true;

	if (fscanf (in, "%d %d %d %d", &dimensions, &vertices, &regions , &n) != 4) 
		{
			fputs ("How many points and regions?\n", stderr);
			bFileReadOkay = false;
		}
	else if (dimensions != 3)
		{
			fputs ("Dimensions should be three!\n", stderr);
			bFileReadOkay = false;
		}
	else if (vertices <= 0)
		{
			fputs ("Too few vertices!\n", stderr);
			bFileReadOkay = false;
		}
	else if (regions <= 0)
		{
			fputs ("Too few regions!\n", stderr);
			bFileReadOkay = false;
		}
	else if (n != 1)
		{
			fputs ("Unexpected value of n!\n", stderr);
			bFileReadOkay = false;
		}
	if (!bFileReadOkay)
		{
			fclose (in);
			return -1;
		}

	V = new Coordinate[vertices];
		
	for (int i = 0; i < vertices; i++)
		{   
			if (fscanf (in, "%lf %lf %lf", &(V[i].x), &(V[i].y), &(V[i].z)) != 3)
				{	
					fputs ("Failed to read vertex points!\n", stderr);
					bFileReadOkay = false;
					break;
				}
			V[i].check_internal ();
		}
	if (!bFileReadOkay)
		{
			fclose (in);
			return -1;
		}
		
	R = new Region[regions];
	
	for (int j = 0; j < regions; j++)
		{ 
			int subregion = 0;

			if (fscanf (in, "%d", &subregion) != 1)
				{
					fputs ("Failed to read number of points in next region!\n", stderr);
					bFileReadOkay = false;
					break;
				}
			else if (subregion < 4)
				{
					fputs ("Too few points in next region!\n", stderr);
					bFileReadOkay = false;
					break;
				}
			for (int k =0; k < subregion; k++)
				{	
					int num = 0;

					if (fscanf (in, "%d", &num) != 1)
						{
							fputs ("Failed to read point in current region!\n", stderr);
							bFileReadOkay = false;
							break;
						}
					R[j].vertex.push_back (num); 
				}
			if (!bFileReadOkay) break;

			for (int l = 0; l < R[j].vertex.size (); l++)
				{ 
					R[j].Centre.x = R[j].Centre.x + V[(R[j].vertex[l])].x;
					R[j].Centre.y = R[j].Centre.y + V[(R[j].vertex[l])].y;
					R[j].Centre.z = R[j].Centre.z + V[(R[j].vertex[l])].z;
				}
			R[j].Centre.x = R[j].Centre.x / R[j].vertex.size ();
			R[j].Centre.y = R[j].Centre.y / R[j].vertex.size ();
			R[j].Centre.z = R[j].Centre.z / R[j].vertex.size ();
	
    		//std:: cout << R[j].Centre.x << " " << R[j].Centre.y << " " << R[j].Centre.z << "\n";
    		
    		int  s=0;
    		double distance=0;
    		
			for (int g=-1;g<=1;g++)
				for (int  k=-1;k<=1;k++)
					for (int m=-1;m<=1;m++)
						for (int i = 0; i < points; i++)
							{
								Coordinate point;
	
								point.x = C[i].x + m;
								point.y = C[i].y + k;
								point.z = C[i].z + g;
																
								if(	s == 0 )
								{
									distance = R[j].distance_from_centre ( point ); 
									R[j].point = s;
								}
								else
								{
									double d = R[j].distance_from_centre ( point );
									if ( d < distance )
									{
										distance = d;
										R[j].point = s;
									}
								}							
								s++;
							}

			if (R[j].check_internal (V))
				{
				 	//std::cout << "Region " << j << " is inside of box; belongs to point " << R[j].point << "\n";
				}
		}
		
		
	if (!bFileReadOkay)
		{
			fclose (in);
			return -1;
		}

	fclose (in); // close the file we are reading from

	return 0;
}

int Generator::generate_xml	(char * filename_xml)
{
 	FILE * out_xml  = fopen (filename_xml, "w"); // open xml file for writing
 	
 	if (!out_xml)
		{
			fprintf (stderr, "Failed to open xml file \"%s\" for writing!\n", filename_xml);
			return -1;
		}
	
	fprintf (out_xml, "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n<Voronoi-set>\n" " <Points count =\"%d\">\n",points);
	
	int count = 0;
	
	while (count < points)
	{	
		fprintf (out_xml, "  <point id=\"p%d\" x=\"%.16lf\" y=\"%.16lf\" z=\"%.16lf\"/>\n",(13*points+count),C[count].x, C[count].y, C[count].z);
		++count	;
	};
		
	fprintf (out_xml," </Points>\n");
	fprintf (out_xml," <Vertices count =\"%d\">\n",vertices);
	for (int b = 0; b < (int)(points*27); b++)
	//	for (int b = (int)((points*27-points)/2); b < (int)((points*27+points)/2); b++)
		{
			for ( int w = 0 ; w < R[b].vertex.size(); w++)
				{
					V[(R[b].vertex[w])].bVertex = true;
				}
		}
				 
	char * s = 0;
	for (int  j= 0; j < vertices; j++)
		{	
			if (V[j].bInternal == true)
				s = "yes";
			else
				s = "no";
			if (V[j].bVertex == true)
				fprintf (out_xml,"  <Vertex id=\"%d\" x=\"%.16lf\" y=\"%.16lf\" z=\"%.16lf\" internal=\"%s\"/>\n",j,V[j].x, V[j].y, V[j].z,s);
		}
	
	fprintf (out_xml," </Vertices>\n <Regions count=\"%d\">\n",points*27);/////// note no of regions have been changed 31-7-2008

	
	char *r;
		for (int b = 0; b < (int)(points*27); b++)
	//	for (int b = (int)((points*27-points)/2); b < (int)((points*27+points)/2); b++)
		{
			if (R[b].bRegionIsInternal == true)
				r = "yes";
			else
				r = "no";
			if(R[b].bRegionIsInternal == true)
				{
					
					fprintf (out_xml, "  <Region point=\"p%d\" vertices=\"%d\" facets=\"%d\" internal=\"%s\">\n",b,R[b].vertex.size(),R[b].facets,r );
					for( int i = 0; i < R[b].facets; i++)
						{
							fprintf (out_xml, "   <Facet vertices=\"%d\">", R[b].F[i].vertex.size());
							for ( int k=0; k < R[b].F[i].vertex.size(); k++)
								fprintf (out_xml,"%d\t",R[b].F[i].vertex[k]);
							fprintf (out_xml, "</Facet>\n");
						}
					
					
					fprintf (out_xml,"  </Region>\n");
				}
		}
	
	fprintf (out_xml," </Regions>\n</Voronoi-set>");
	
	fclose (out_xml);  // closing xml  output file
}

int Generator::read_regions (char * filename_rbox) // To read regions and store there vertices to file
{
		for (int b = 0; b < (int)(points*27); b++)
	//	for (int b = (int)((points*27-points)/2); b < (int)((points*27+points)/2); b++)
		{	
			FILE * out_rbox  = fopen (filename_rbox, "w");

			fprintf (out_rbox, "3 rbox %d\n%d\n",R[b].vertex.size(),R[b].vertex.size());
		
			for ( int w = 0 ; w < R[b].vertex.size(); w++)
				{
					fprintf ( out_rbox," %.16lf %.16lf %.16lf\n",V[(R[b].vertex[w])].x,V[(R[b].vertex[w])].y,V[(R[b].vertex[w])].z);
				}
			fclose(out_rbox);

			char buf[256 + strlen (filename_rbox)];
			sprintf (buf, "cat %s | qconvex o TO results.txt", filename_rbox);
			system (buf);
			
			char * filename_input = "results.txt";
			FILE * in = fopen (filename_input, "r"); // opening a file for reading
   
    		if (!in) // checking it has opened okay
				{
					fputs ("Failed to open file for reading!\n", stderr);
					return -1;
				}
			
							
			int region_vertices,region_facets,p,dimensions;
			
			fscanf ( in,"%d %d %d %d",&dimensions ,&region_vertices ,&region_facets ,&p);
			
			R[b].create_facets(region_facets);
									
			for (int p=0; p < region_vertices; p++)
				{
					double tmp_x,tmp_y,tmp_z;
					fscanf (in,"%lf %lf %lf", &tmp_x,&tmp_y,&tmp_z);// NOT WORKING FINE......WHY??
					std::cout<<tmp_x<<" "<<tmp_y<<" "<<tmp_z<<"\n";
				}
													 	
			for (int j=0; j < region_facets; j++)
				{	
					int subfacet=0;
					fscanf (in,"%d",&subfacet);
					//std::cout<<subfacet;
										
					for ( int k=0; k < subfacet; k++)
						{
							int num = 0;
							fscanf (in,"%d",&num);
							R[b].F[j].vertex.push_back(R[b].vertex[num]);
							
						}
				}
			fclose(in);
			// read results.txt: number of facets?
			// R[b].create_facets (??);
			// vertex j in output file is really R[b].vertex[j]
		
		}
	return 0;
}	

